/*
 * Copyright (c) 2001 Red Hat, Inc.
 *           (c) 2010 One Laptop Per Child
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Red Hat not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  Red Hat makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * RED HAT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL RED HAT
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Authors:  Owen Taylor, Red Hat, Inc.
 *           Sayamindu Dasgupta <sayamindu@laptop.org>
 */

#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <gconf/gconf-client.h>

#include "xsettings-manager.h"

#define SUGAR_FONT_DIR "/desktop/sugar/font"
#define SUGAR_FONT_FACE_KEY "/desktop/sugar/font/default_face"
#define SUGAR_FONT_SIZE_KEY "/desktop/sugar/font/default_size"

/* See http://freedesktop.org/wiki/Specifications/XSettingsRegistry */
#define XSETTINGS_REGISTRY_GTK_FONT_NAME "Gtk/FontName"

XSettingsManager *manager;
GConfClient *client;

void terminate_cb()
{
    gtk_main_quit();
}

GdkFilterReturn
manager_event_filter(GdkXEvent * xevent, GdkEvent * event, gpointer data)
{
    if (xsettings_manager_process_event(manager, (XEvent *) xevent))
	return GDK_FILTER_REMOVE;
    else
	return GDK_FILTER_CONTINUE;
}

void update_font()
{
    gchar *font_face, *font_name;
    gdouble font_size;

    font_face = gconf_client_get_string(client, SUGAR_FONT_FACE_KEY, NULL);
    if (!font_face)
	return;

    font_size = gconf_client_get_float(client, SUGAR_FONT_SIZE_KEY, NULL);
    if (!font_size)
	return;

    font_name = g_strdup_printf("%s %f", font_face, font_size);

    xsettings_manager_set_string(manager, XSETTINGS_REGISTRY_GTK_FONT_NAME,
				 font_name);
    xsettings_manager_notify(manager);
}

void
font_face_changed_callback(GConfClient * client,
			   guint cnxn_id,
			   GConfEntry * entry, gpointer user_data)
{
    if (entry->value == NULL) {
	return;
    } else {
	if (entry->value->type == GCONF_VALUE_STRING) {
	    update_font();
	} else {
	    return;
	}
    }
}

void
font_size_changed_callback(GConfClient * client,
			   guint cnxn_id,
			   GConfEntry * entry, gpointer user_data)
{
    if (entry->value == NULL) {
	return;
    } else {
	if (entry->value->type == GCONF_VALUE_FLOAT) {
	    update_font();
	} else {
	    return;
	}
    }
}


void setup_font()
{
    client = gconf_client_get_default();
    gconf_client_add_dir(client,
			 SUGAR_FONT_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);

    gconf_client_notify_add(client, SUGAR_FONT_FACE_KEY,
			    (GConfClientNotifyFunc)
			    font_face_changed_callback, NULL, NULL, NULL);

    gconf_client_notify_add(client, SUGAR_FONT_SIZE_KEY,
			    (GConfClientNotifyFunc)
			    font_size_changed_callback, NULL, NULL, NULL);

    update_font();
}

int main(int argc, char **argv)
{
    gtk_init(&argc, &argv);

    if (xsettings_manager_check_running
	(gdk_display, DefaultScreen(gdk_display))) {
	fprintf(stderr,
		"Could not create manager! Manager already running.\n");
	exit(1);
    }

    manager =
	xsettings_manager_new(gdk_display, DefaultScreen(gdk_display),
			      terminate_cb, NULL);
    if (!manager) {
	fprintf(stderr, "Could not create manager!\n");
	exit(1);
    }

    gdk_window_add_filter(NULL, manager_event_filter, NULL);

    setup_font();
    gtk_main();

    xsettings_manager_destroy(manager);
    g_object_unref(client);

    return 0;
}
