/*
 * Copyright (C) 2006 Dirk Mueller <mueller@kde.org>
 * Copyright (C) 2006 Zack Rusin <zack@kde.org>
 * Copyright (C) 2006 Simon Hausmann <hausmann@kde.org>
 * Copyright (C) 2007 Ryan Leavengood <leavengood@gmail.com>
 * Copyright (C) 2008 Andrea Anzani <andrea.anzani@gmail.com>
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//#define DIRECT_ENABLE_DEBUG

#include "config.h"
#include "Image.h"

#include "BitmapImage.h"
#include "FloatRect.h"
#include "GraphicsContext.h"
#include "SharedBuffer.h"
#include "NotImplemented.h"
#include "PlatformString.h"

#include <directfb.h>

extern "C" {
#include <direct/debug.h>
#include <direct/messages.h>
}


D_DEBUG_DOMAIN( WebKit_Image, "WebKit/Image", "WebKit Image" );

/**********************************************************************************************************************/

// This function loads resources from WebKit
Vector<char> loadResourceIntoArray(const char*);


namespace WebCore {

bool FrameData::clear(bool clearMetadata)
{
    if (clearMetadata)
        m_haveMetadata = false;

    if (m_frame) {
        m_frame->Release( m_frame );
        m_frame = 0;
        m_duration = 0.0f;
        m_hasAlpha = true;
        return true;
    }

    return false;
}

WTF::PassRefPtr<Image> Image::loadPlatformResource(const char* name)
{
    Vector<char> array = loadResourceIntoArray(name);
    WTF::PassRefPtr<BitmapImage> image = BitmapImage::create();
    RefPtr<SharedBuffer> buffer = SharedBuffer::create(array.data(), array.size());
    image->setData(buffer, true);

    return image;
}

void BitmapImage::initPlatformData()
{
}

void BitmapImage::invalidatePlatformData()
{
}

// Drawing Routines
void BitmapImage::draw(GraphicsContext* ctxt, const FloatRect& dst, const FloatRect& src, ColorSpace styleColorSpace, CompositeOperator op)
{
    D_DEBUG_AT( WebKit_Image, "%s( %4d,%4d-%4dx%4d <- %4d,%4d-%4dx%4d )\n", __FUNCTION__,
                (int) dst.x(), (int) dst.y(), (int) dst.width(), (int) dst.height(),
                (int) src.x(), (int) src.y(), (int) src.width(), (int) src.height() );

    startAnimation();

    IDirectFBSurface* image = nativeImageForCurrentFrame();
    if (!image) // If the image hasn't fully loaded.
        return;

    if (mayFillWithSolidColor()) {
        fillWithSolidColor(ctxt, dst, solidColor(), styleColorSpace, op);
        return;
    }

    ctxt->save();
    ctxt->setCompositeOperation(op);

    DFBRectangle srcRect = { src.x(), src.y(), src.width(), src.height() };
    DFBRectangle dstRect = { dst.x(), dst.y(), dst.width(), dst.height() };

    // Test using example site at
    // http://www.meyerweb.com/eric/css/edge/complexspiral/demo.html

    ctxt->platformContext()->StretchBlit(ctxt->platformContext(), image, &srcRect, &dstRect);
    ctxt->restore();
}

void Image::drawPattern(GraphicsContext* context, const FloatRect& tileRect, const AffineTransform& patternTransform, const FloatPoint& srcPoint, ColorSpace, CompositeOperator op, const FloatRect& dstRect)
{
    D_DEBUG_AT( WebKit_Image, "%s( %4d,%4d-%4dx%4d <- %4d,%4d )\n", __FUNCTION__,
                (int) dstRect.x(), (int) dstRect.y(), (int) dstRect.width(), (int) dstRect.height(),
                (int) srcPoint.x(), (int) srcPoint.y() );

    // FIXME: finish this to support also phased position (srcPoint)
    startAnimation();

    IDirectFBSurface* image = nativeImageForCurrentFrame();
    if (!image) // If the image hasn't fully loaded.
        return;

//    image->Dump( image, "/", "pattern" );

    float currentW = 0;
    float currentH = 0;

    context->save();
    context->setCompositeOperation(op);
    context->clip(enclosingIntRect(dstRect));

    while (currentW < dstRect.width()) {
        while (currentH < dstRect.height()) {
            context->platformContext()->Blit(context->platformContext(), image, NULL, dstRect.x() + currentW, dstRect.y() + currentH);
            currentH += tileRect.height();
        }
        currentW += tileRect.width();
        currentH = 0;
    }
    context->restore();
}

void BitmapImage::checkForSolidColor()
{
    // FIXME: need to check the RGBA32 buffer to see if it is 1x1.
    m_isSolidColor = false;
    m_checkedForSolidColor = true;
}

IDirectFBSurface* BitmapImage::getDirectFBSurface() const
{
    return const_cast<BitmapImage*>(this)->frameAtIndex(0);
}

} // namespace WebCore

