# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from gettext import gettext as _

import gtk
import wnck
import gconf
import gobject

import zerosugar
import sugar_network

from sugar import wm
from sugar.graphics.xocolor import XoColor
from sugar.activity.activityfactory import create_activity_id

from jarabe.view.launcher import LaunchWindow
from jarabe.model.bundleregistry import get_registry
from jarabe.journal import model
from jarabe.plugins.sn import client


_logger = logging.getLogger('plugins.sn_browser.launcher')


class Launcher(object):

    def __init__(self):
        self._launches = {}
        self._screen = wnck.screen_get_default()
        self._screen.connect('window-opened', self.__window_opened_cb)
        client.connect('launch', self.__launch_cb)

    def launch(self, bundle, activity_id=None, object_id=None, uri=None,
            color=None, invited=None, args=None):

        def found_jobject(props):
            self._launch(bundle,
                    props.get('activity_id') or activity_id,
                    props.get('object_id') or object_id,
                    uri,
                    XoColor(props['icon-color']) if 'icon-color' in props \
                            else color,
                    args)

        def not_found_jobject(error):
            _logger.exception('Failed to launch %r: %s',
                    bundle.get_bundle_id(), error)

        if activity_id and not object_id:
            _logger.debug('Look for jobject for %r activity_id', activity_id)
            model._get_datastore().find({'activity_id': activity_id}, ['uid'],
                    reply_handler=lambda jobjects, total: \
                            found_jobject(jobjects[0] if total else {}),
                    error_handler=not_found_jobject, byte_arrays=True)
        elif object_id and not activity_id:
            _logger.debug('Look for %r jobject', object_id)
            model._get_datastore().get_properties(object_id,
                    reply_handler=found_jobject,
                    error_handler=not_found_jobject, byte_arrays=True)
        else:
            self._launch(bundle, activity_id, object_id, uri, color, args)

    def _launch(self, bundle, activity_id, object_id, uri, color, args):
        if not activity_id:
            activity_id = create_activity_id()
        if color is None:
            gc = gconf.client_get_default()
            color = XoColor(gc.get_string('/desktop/sugar/user/color'))

        if args is None:
            args = []
        if activity_id:
            args.extend(['-a', activity_id])
        if object_id:
            args.extend(['-o', object_id])
        if uri:
            args.extend(['-u', uri])

        _logger.info('Starting %r: activity_id=%r object_id=%r uri=%r',
                bundle.get_bundle_id(), activity_id, object_id, uri)

        pipe = zerosugar.launch('/', bundle.get_bundle_id(), 'activity', args)
        gobject.io_add_watch(pipe.fileno(), gobject.IO_IN | gobject.IO_HUP,
                self.__progress_cb, pipe, activity_id)

        window = LaunchWindow(activity_id, bundle.get_icon(), color)
        window.connect('realize', self.__window_realize_cb,
                bundle.get_bundle_id(), activity_id)
        window.show()
        self._launches[activity_id] = window

    def _stop_launcher(self, activity_id):
        if activity_id not in self._launches:
            return
        _logger.debug('Stop %r launcher', activity_id)
        window = self._launches.pop(activity_id)
        window.destroy()

    def _failure_report(self, impl, log_path):
        import sugar_network

        online = sugar_network.Client('/')
        offline = sugar_network.Client('~')

        cls = online.Report if online.connected else offline.Report
        report = cls(implementation=impl or '', description='foo')
        report.post()
        if log_path:
            report.upload_blob('data', log_path)

    def __window_opened_cb(self, screen, window):
        if window.get_window_type() != wnck.WINDOW_NORMAL or \
                wm.get_sugar_window_type(window) == 'launcher':
            return
        activity_id = wm.get_activity_id(window)
        if activity_id:
            self._stop_launcher(activity_id)

    def __window_realize_cb(self, widget, bundle_id, activity_id):
        wm.set_activity_id(widget.window, str(activity_id))
        widget.window.property_change('_SUGAR_WINDOW_TYPE', 'STRING', 8,
                gtk.gdk.PROP_MODE_REPLACE, 'launcher')
        wm.set_bundle_id(widget.window, str(bundle_id))

    def __launch_cb(self, sender, bundle_id, command, object_id, uri, args):
        bundle = get_registry().get_bundle(bundle_id, True)
        self.launch(bundle, object_id=object_id, uri=uri, args=args)

    def __progress_cb(self, source, cb_condition, pipe, activity_id):
        event = pipe.read()
        _logger.debug('Execution progress for %r: %r',
                activity_id, event or 'exit')
        if event is None:
            self._stop_launcher(activity_id)
            return False

        phase, props = event
        try:
            if phase == 'failure':
                _logger.warning('Activity %r failed', activity_id)
                self._failure_report(props.get('implementation'),
                        props.get('log_path'))
        except Exception:
            _logger.exception('Failed to process event')

        return True
