# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from gettext import gettext as _

import gtk
import glib
import webkit
import gobject

from sugar import wm
from sugar.graphics.window import Window
from sugar.activity.activityfactory import create_activity_id

from jarabe.view.palettes import BasePalette
from jarabe.plugins.sn import client
from jarabe.plugins.sn_browser import SN_BROWSER_NAME, SN_APP_URL


_WEBKIT_LOAD_PROVISIONAL = 0
_WEBKIT_LOAD_FINISHED = 2
_WEBKIT_LOAD_FAILED = 4

_RETRY_TIMEOUT = 1

_logger = logging.getLogger('plugins.sn_browser.browser')


class Browser(Window):

    def __init__(self):
        Window.__init__(self)
        self._initial_load = True

        self.set_title(_('Sugar Network'))
        self.connect('realize', self.__realize_cb)
        self.iconify()

        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrolled.set_shadow_type(gtk.SHADOW_NONE)
        self.set_canvas(scrolled)

        self._webkit = webkit.WebView()
        scrolled.add(self._webkit)

        self.show_all()

        self._webkit.connect('notify::load-status', self.__load_status_cb)
        client.connect('connect', self.__connection_cb)

        gobject.timeout_add_seconds(_RETRY_TIMEOUT, self._open)

    def _open(self):
        _logger.debug('Open %s', SN_APP_URL)
        self._webkit.open(SN_APP_URL)

    def __load_status_cb(self, widget, pspec):
        _logger.debug('Webkit status changed to %r',
                self._webkit.props.load_status)
        if self._webkit.props.load_status == _WEBKIT_LOAD_FINISHED:
            self._initial_load = False
            self.get_window().set_cursor(None)
        elif self._webkit.props.load_status == _WEBKIT_LOAD_FAILED:
            if self._initial_load:
                gobject.timeout_add_seconds(_RETRY_TIMEOUT, self._open)
            else:
                self.get_window().set_cursor(None)
        elif self._webkit.props.load_status == _WEBKIT_LOAD_PROVISIONAL:
            self.get_window().set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))

    def __realize_cb(self, window):
        wm.set_bundle_id(window.window, SN_BROWSER_NAME)
        wm.set_activity_id(window.window, create_activity_id())
        self.disconnect_by_func(self.__realize_cb)

    def __connection_cb(self, sender, mountpoint, connected):
        self._open()


class Palette(BasePalette):

    def __init__(self, activity):
        self._activity = activity
        BasePalette.__init__(self, activity)

    def setup_palette(self):
        title = self._activity.get_title()
        self.set_primary_text(glib.markup_escape_text(title))
