# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import logging
from os.path import lexists, basename, join
from gettext import gettext as _

import gtk
import gobject

import sugar_network
from sugar.util import LRU
from sugar.bundle.activitybundle import ActivityBundle

from jarabe.plugins.sn_browser import SN_BROWSER_NAME
from jarabe.plugins.sn import client


_logger = logging.getLogger('plugins.sn.bundleregistry')
_stub_icon_path = None
_online_cache = LRU(100)


class BundleRegistry(gobject.GObject):
    """Tracks the available activity bundles"""

    __gsignals__ = {
        'bundle-added': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                         ([gobject.TYPE_PYOBJECT])),
        'bundle-removed': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                           ([gobject.TYPE_PYOBJECT])),
        'bundle-changed': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                           ([gobject.TYPE_PYOBJECT])),
    }

    def __init__(self):
        gobject.GObject.__init__(self)
        self._bundles = {}

        for props in client.find('~', 'context', 0, 50,
                reply=['guid', 'keep', 'keep_impl', 'position']):
            if props['keep_impl']:
                self._add_bundle(props['guid'], props)

        client.connect('keep', self.__keep_cb)
        client.connect('keep_impl', self.__keep_impl_cb)

    def __iter__(self):
        return self._bundles.itervalues()

    def __len__(self):
        return len(self._bundles)

    def get_bundle(self, context_guid, is_online=False):
        if context_guid == 'org.laptop.JournalActivity':
            return None
        elif context_guid == SN_BROWSER_NAME:
            return _BrowserInfo()

        if context_guid in self._bundles:
            return self._bundles[context_guid]

        if context_guid in _online_cache:
            return _online_cache[context_guid]

        if not is_online:
            return None

        try:
            props = client.get('/', 'context', context_guid,
                    reply=['guid', 'keep', 'keep_impl', 'title'])
            result = _ContextInfo(props)
        except Exception, error:
            _logger.warning(_('Cannot fetch activity info: %s'), error)
            result = None

        _online_cache[context_guid] = result

        return result

    def get_activities_for_type(self, mime_type):
        # TODO
        return []

    def is_installed(self, bundle):
        return hasattr(bundle, 'get_bundle_id') and \
                bundle.get_bundle_id() in self._bundles

    def is_bundle_favorite(self, bundle_id, version):
        bundle = self._bundles.get(bundle_id)
        if bundle is None:
            return False
        return bundle.props['keep']

    def set_bundle_favorite(self, bundle_id, version, keep):
        bundle = self._bundles.get(bundle_id)
        if bundle is None:
            return
        client.update('~', 'context', bundle_id, keep=keep)
        bundle.props['keep'] = keep
        self.emit('bundle-changed', bundle)

    def get_bundle_position(self, bundle_id, version):
        bundle = self._bundles.get(bundle_id)
        if bundle is None:
            return -1, -1
        return bundle.props['position']

    def set_bundle_position(self, bundle_id, version, x, y):
        bundle = self._bundles.get(bundle_id)
        if bundle is None:
            return
        position = bundle.props['position'] = [int(x), int(y)]
        client.update('~', 'context', bundle_id, position=position)
        self.emit('bundle-changed', bundle)

    def get_default_for_type(self, mime_type):
        # TODO
        pass

    def is_activity_protected(self, bundle_id):
        # TODO
        return False

    def install(self, bundle, uid=None, force_downgrade=False):
        raise NotImplementedError('Not yet supported')

    def uninstall(self, bundle, force=False, delete_profile=False):
        raise NotImplementedError('Not yet supported')

    def upgrade(self, bundle):
        raise NotImplementedError('Not yet supported')

    def _add_bundle(self, bundle_id, props):
        for path in sugar_network.checkins(bundle_id):
            try:
                bundle = self._bundles[bundle_id] = \
                        _BundleInfo(ActivityBundle(path), props)
                _logger.info('Add %r bundle', bundle_id)
                return bundle
            except Exception, error:
                _logger.exception('Cannot load %r bundle from %r',
                        bundle_id, path)
        else:
            _logger.info('No bundles for %r', bundle_id)

    def __keep_cb(self, sender, bundle_id, keep):
        bundle = self._bundles.get(bundle_id)
        if bundle is None:
            return
        bundle.props['keep'] = keep
        self.emit('bundle-changed', bundle)

    def __keep_impl_cb(self, sender, bundle_id, keep_impl):
        if keep_impl:
            props = client.get('~', 'context', bundle_id,
                    reply=['guid', 'keep', 'keep_impl', 'position'])
            bundle = self._bundles.get(bundle_id)
            if bundle is None:
                bundle = self._add_bundle(bundle_id, props)
            else:
                bundle.props.update(props)
            self.emit('bundle-added', bundle)
        else:
            if bundle_id in self._bundles:
                bundle = self._bundles.pop(bundle_id)
                self.emit('bundle-removed', bundle)


class _ContextInfo(object):

    def __init__(self, props):
        self.props = props

    def get_name(self):
        return self.props['title']

    def get_bundle_id(self):
        return self.props['guid']

    def get_icon(self):
        path = None
        try:
            path, __ = client.get_blob_path('/', 'context',
                    self.get_bundle_id(), 'artifact_icon')
        except Exception, error:
            _logger.debug('Fail to get icon for %r: %s',
                    self.get_bundle_id(), error)

        if not path or os.stat(path).st_size == 0:
            return _stub_icon()
        else:
            svg_path = path + '.svg'
            if not lexists(svg_path):
                os.symlink(basename(path), svg_path)
            return svg_path

    def get_tags(self):
        # Doesn't matter with Sweets' features enabled
        return []

    def get_activity_version(self):
        return ''

    def get_installation_time(self):
        return 0

    def get_command(self):
        # Doesn't matter with Sweets' features enabled
        return ''

    def is_user_activity(self):
        # Doesn't matter with Sweets' features enabled
        return False

    def get_path(self):
        return '/'


class _BundleInfo(object):

    def __init__(self,  bundle, props):
        self.props = props
        self._bundle = bundle

    def __getattr__(self, name):
        return getattr(self._bundle, name)


class _BrowserInfo(object):

    def __init__(self):
        icon = gtk.icon_theme_get_default().lookup_icon('sugar-network', 0, 0)
        self._icon_path = icon.get_filename()

    def get_name(self):
        return _('Sugar Network')

    def get_bundle_id(self):
        return SN_BROWSER_NAME

    def get_icon(self):
        return self._icon_path

    def get_tags(self):
        return []

    def get_activity_version(self):
        return '0'

    def get_installation_time(self):
        return 0

    def get_command(self):
        return ''

    def is_user_activity(self):
        return False

    def get_path(self):
        return '/'


def _stub_icon():
    global _stub_icon_path

    if not _stub_icon_path:
        theme = gtk.icon_theme_get_default()
        _stub_icon_path = theme.lookup_icon('empty', 0, 0).get_filename()
    return _stub_icon_path
