#!/usr/bin/env python

# Copyright (C) 2007  Phyrum Tea <phyrum.tea@sign.ch>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import pygtk
pygtk.require('2.0')
import gtk
import gtk.glade
import time
import gobject
import pango
import logging
import locale
import gettext
from gettext import gettext as _
from math import pi, sin, sqrt
from wordsearch import *

class WordsearchApp:
  def __init__(self, window, base_dir = None):
    if base_dir is None:
      base_dir = '.'
    locale.setlocale(locale.LC_ALL, '')
    gettext.bindtextdomain('wordsearch', base_dir + '/locale')
    gettext.textdomain('wordsearch')

    dic = {"on_quit" : gtk.main_quit
         , "on_new" : self.on_new
         , "on_open" : self.on_open
         , "on_saveas" : self.on_saveas
         , "on_about" : self.on_about}
    wtree = gtk.glade.XML(base_dir + '/wordsearch.glade')
    wtree.signal_autoconnect(dic)
    if window is None:
      window = wtree.get_widget("main_wnd")
      window.connect("destroy", gtk.main_quit)
      window.set_default_size(800, 600)
      window.show()
      sw = wtree.get_widget("scrolledwindow")
    else:
      widget = window.get_children()
      sw = gtk.ScrolledWindow()
      if len(widget) > 0:
        widget[0].add(sw)
      sw.show()
    
    self.sw = sw
    self.base_dir = base_dir
    self.wtree = wtree

  def compile(self, rows, cols, filename):
    self.createWordsearch(filename)
    area = WordsearchArea()
    area.set_wordsearch(self.wordsearch)
    area.connect('wordsearch_finished', self._wordsearchFinished)
    area.show()
    self.sw.add_with_viewport(area)

  def createWordsearch(self, file):
    f = open(file)
    words = []

    try:
      for line in f:
        words.append(line.strip().upper())
    finally:
      f.close()

    wc = WordsearchCompiler(10, 15)
    self.wordsearch = wc.create(words)

  def on_about(self, widge):
    dlg = self.wtree.get_widget("about_dialog")
    dlg.set_name(_("Sugar Wordsearch"))
    dlg.run()
    dlg.hide()

  def on_new(self, widget):
    dlg = self.wtree.get_widget("new_dialog")
    dlg.set_size_request(500, 400)
    response = dlg.run()
    if response == gtk.RESPONSE_OK:
      self._extractWords("This program is free software.")
      children = dlg.get_children()
      child = self.wtree.get_widget("textview1")
      self.compile(10, 15, "wordlist/sample.txt")
      
    dlg.hide()

  def on_open(self, widge):
    dlg = self.wtree.get_widget("filechooserdlg")
    dlg.set_action(gtk.FILE_CHOOSER_ACTION_OPEN)
    dlg.set_title(_("Open"))
    response = dlg.run()
    dlg.hide()

  def on_saveas(self, widget):
    dlg = self.wtree.get_widget("filechooserdlg")
    dlg.set_action(gtk.FILE_CHOOSER_ACTION_SAVE)
    dlg.set_title(_("Save As"))
    response = dlg.run()
    dlg.hide()

  def _wordsearchFinished(self, widget, event, jkl):
    # TODO Show popup dialog instead modal dialog.
    dlg = self.wtree.get_widget("finished_dialog")
    dlg.run()
    dlg.hide()

  def _extractWords(self, text):
    for s in text.split(' '):
      ss = s.strip(' \t,.;').upper()
      print ss

class WordsearchArea(gtk.DrawingArea):
  'Reusable GTK Wordsearch Widget'
  __gsignals__ = {
      'wordsearch_finished' : (gobject.SIGNAL_RUN_FIRST, None, [int, gobject.TYPE_PYOBJECT])
  }

  def __init__(self):
    gtk.DrawingArea.__init__(self)
    self.wordsearch = None
    self.field_width = 35
    self._mouseCurrent = None
    self.point_a = None
    self.point_b = None
    self.mousePressed = False
    self.status = None
    self._animations = []
    self.insets = {"left" : 20, "top" : 20, "right" : 20, "bottom" : 20}
    self.add_events(gtk.gdk.POINTER_MOTION_MASK | gtk.gdk.POINTER_MOTION_HINT_MASK
      | gtk.gdk.KEY_PRESS_MASK | gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK
      | gtk.gdk.DRAG_ENTER | gtk.gdk.SCROLL_MASK)
    self.connect("expose-event", self.expose)
    self.connect("button_press_event", self._mousePressed)
    self.connect("button_release_event", self._mouseReleased)
    #super(gtk.DrawingArea, self).connect("motion_notify_event", self._mouseMoved)
    self._layout = self.create_pango_layout("")
    self.modify_bg(gtk.STATE_NORMAL, gtk.gdk.Color(65535, 65535, 65535))
    self.revealed_color = [1, 0, 0, 0.3]
    self.guessed_color = [0, 1, 0, 0.3]
    gobject.timeout_add(100, self._timer)

#  def do_realize(self):
#    print "do_realize"
#
#  def do_size_request(self, request):
#    print "do_size_request"
#    width, height = self._layout.get_size()
#    request.width = width
#    request.height = height
#
#  def do_size_allocate(self, alloc):
#    print "do_size_allocate"
#    self.alloc = alloc

  # TODO Instead of drawing everything, draw only dirty areas.
  def expose(self, widget, event):
    context = widget.window.cairo_create()
    if self.wordsearch != None:
      self._drawLetters(context)
      self._drawWordlist(context)
      self._drawCandidates(context)
      if self.point_b is None:
        self._drawStatus(context)

    for a in self._animations:
      a.draw(context, self._layout)
      if a.finished():
        self._animations.remove(a)

    return True

  def set_wordsearch(self, wordsearch):
    self.wordsearch = wordsearch
    x = self.insets["left"] + 25 + self.wordsearch.cols*self.field_width
    timer = TimeAnimation(x, 500)
    self._animations.append(timer)
    timer.start()
    self.timer = timer

  def _drawLetters(self, context):
    layout = self._layout
    fields = self.wordsearch.fields
    fw = self.field_width
    
    context.set_source_rgb(0, 0, 0)
    i = 0
    for r in fields:
      j = 0
      for c in r:
        layout.set_text(c.char)
        layout.set_font_description(pango.FontDescription("Sans 16"))
        off_x = (fw - layout.get_pixel_size()[0] + 1)/2
        off_y = (fw - layout.get_pixel_size()[1] + 1)/2
        context.move_to(self.insets["left"]+j*fw + off_x, self.insets["top"]+i*fw + off_y)
        context.update_layout(layout)
        context.show_layout(layout)
        j += 1
      i += 1

  def _drawCandidates(self, context):
    
    for w in self.wordsearch.words:
      if w.status == Candidate.STATUS_REVEALED:
        color = self.revealed_color
      elif w.status == Candidate.STATUS_GUESSED:
        color = self.guessed_color
      else:
        continue

      x = self.insets["left"]+w.col*self.field_width+self.field_width/2
      y = self.insets["top"]+w.row*self.field_width+self.field_width/2
      r = self.field_width*0.9/2

      mx, my = x, y
      if w.dir == Candidate.HORIZONTAL:
        a1, a2 = pi*3/2, pi/2
        wc = w.col + len(w.word)-1
        wr = w.row
        mx, my = x, y - r
      elif w.dir == Candidate.VERTICAL:
        a1, a2 = 0, pi
        wc = w.col
        wr = w.row + len(w.word)-1
        mx, my = x + r, y
      elif w.dir == Candidate.DOWNDIAGONAL:
        a1, a2 = pi*7/4, pi*3/4
        wc = w.col + len(w.word)-1
        wr = w.row + len(w.word)-1
        mx, my = x + r*sin(pi/4), y - r*sin(pi/4)
      elif w.dir == Candidate.UPDIAGONAL:
        a1, a2 = pi*5/4, pi*1/4
        wc = w.col + len(w.word)-1
        wr = w.row - len(w.word)+1
        mx, my = x - r*sin(pi/4), y - r*sin(pi/4)

      x2 = self.insets["left"]+wc*self.field_width+self.field_width/2
      y2 = self.insets["top"]+wr*self.field_width+self.field_width/2

      context.move_to(mx, my)
      context.arc(x2, y2, r, a1, a2)
      context.arc(x, y, r, a2, a1)
      context.set_source_rgba(color[0], color[1], color[2], color[3])
      context.fill_preserve()
      context.set_source_rgb(0, 0, 0)
      context.stroke()

  def _drawWordlist(self, context):
    layout = self._layout
    words = self.wordsearch.words
    i = 0
    for w in words:
      layout.set_text(w.word)
      layout.set_font_description(pango.FontDescription("Sans 16"))
      context.move_to(self.insets["left"] + 25 + self.wordsearch.cols*self.field_width, self.insets["top"]+i*self.field_width)
      context.update_layout(layout)
      context.show_layout(layout)

      if w.status == Candidate.STATUS_REVEALED:
        color = self.revealed_color
      elif w.status == Candidate.STATUS_GUESSED:
        color = self.guessed_color
      else:
        i += 1
        continue

      x = self.insets["left"] + 25 + self.wordsearch.cols*self.field_width
      y = self.insets["top"] + i*self.field_width
      r = layout.get_pixel_size()[1]*0.9/2

      a1, a2 = pi*3/2, pi/2
      wc = w.col + len(w.word)-1
      wr = i
      mx, my = x, y
      
      x2 = self.insets["left"] + 25 + self.wordsearch.cols*self.field_width + layout.get_pixel_size()[0]
      y2 = self.insets["top"] + i*self.field_width

      context.move_to(mx, my)
      context.arc(x2, y2 + r, r, a1, a2)
      context.arc(x, y + r, r, a2, a1)
      context.set_source_rgba(color[0], color[1], color[2], color[3])
      context.fill_preserve()
      context.set_source_rgb(0, 0, 0)
      context.stroke()

      i += 1

  def _drawStatus(self, context):
    if self.status is None:
      return
    fp = self._mouseToField(self.point_a.x, self.point_a.y)
    r = self.field_width/2
    x = self.insets["left"] + fp.x*self.field_width + r
    y = self.insets["top"] + fp.y*self.field_width + r
    
    context.move_to(x + r, y)
    context.arc(x, y, r, 0, 2 * pi)
    context.set_source_rgba(1, 0, 1, 0.3)
    context.fill_preserve()
    context.set_source_rgb(0, 0, 0)
    context.stroke()
    
    x2, y2 = self.get_pointer()

    dr = sqrt((x2 - x)**2 + (y2 - y)**2)
    if dr > 0:
      context.move_to(x+r*(x2-x)/dr, y+r*(y2-y)/dr)
    context.line_to(x2, y2)
    context.stroke()

  def _mouseMoved(self, area, event):
    if self._mouseCurrent is not None:
      x, y = self._mouseCurrent.x, self._mouseCurrent.y
    else:
      x, y = -1, -1
    self._mouseCurrent = Point(event.x, event.y)

    if self.status == "started":
      if x != event.x and y != event.y:
        self.queue_draw()

  def _mousePressed(self, area, event):
    if self._mouseToField(event.x, event.y) is None:
      self.status = None
      self.point_a = None
      self.point_b = None
      return

    if self.status != "started":
      self.point_a = Point(event.x, event.y)
      self.point_b = None
    else:
      self.point_b = Point(event.x, event.y)

  def _mouseReleased(self, area, event):
    if self.point_a is None:
      return
    fw = self.field_width
    if self.status == "started":
      self.status = "finished"
      if not self._select(self.point_a, self.point_b):
        fp = self._mouseToField(self.point_a.x, self.point_a.y)
        x1 = self.insets["left"] + fp.x*fw + fw/2
        y1 = self.insets["top"] + fp.y*fw + fw/2
        fp = self._mouseToField(self.point_b.x, self.point_b.y)
        x2 = self.insets["left"] + fp.x*fw + fw/2
        y2 = self.insets["top"] + fp.y*fw + fw/2
        self._animations.append(SelectAnimation(fw/2, Point(x1, y1), Point(x2, y2)))
      else:
        # check if all words are found
        if len(self.wordsearch.get_normal()) == 0 and len(self.wordsearch.get_hidden()) == 0:
          self.timer.stop()
          self.emit('wordsearch_finished', 0, False)
    else:
      self.status = "started"
    self.queue_draw()

  def _mouseToField(self, x, y):
    """Translates widget coordinates to field coordinates"""
    fx = (x - self.insets["left"])//self.field_width
    fy = (y - self.insets["top"])//self.field_width
    if fx < 0 or fx >= self.wordsearch.cols or fy < 0 or fy >= self.wordsearch.rows:
      return None
    return Point(fx, fy)

  def _select(self, p1, p2):
    f1 = self._mouseToField(p1.x, p1.y)
    f2 = self._mouseToField(p2.x, p2.y)
    if f1 is None or f2 is None: return False

    w = self.wordsearch.get_word(f1.y, f1.x, f2.y, f2.x)
    if w is not None:
      end = len(w.word) - 1
      if w.dir == Candidate.HORIZONTAL:
        wc = w.col + end
        wr = w.row
      elif w.dir == Candidate.VERTICAL:
        wc = w.col
        wr = w.row + end
      elif w.dir == Candidate.DOWNDIAGONAL:
        wc = w.col + end
        wr = w.row + end
      elif w.dir == Candidate.UPDIAGONAL:
        wc = w.col + end
        wr = w.row - end

      if wc == f2.x and wr == f2.y:
        w.status = Candidate.STATUS_GUESSED
        return True
      
      return False
        
  def _animate(self):
    t = time.time()
    for a in self._animations:
      a.frame(t)
    self.queue_draw()

  def _timer(self):
    self._animate()
    return True

#gobject.type_register(WordsearchArea)

class Point:
  def __init__(self, x, y):
    self.x = x
    self.y = y

class SelectAnimation:
  def __init__(self, radius, point_a, point_b):
    self.radius = radius
    self.point_a = point_a
    self.point_b = point_b
    self.alpha = 0.6
  
  def draw(self, context, layout = None):
    if self.alpha <= 0.0: return
    r = self.radius
    x = self.point_a.x
    y = self.point_a.y
    context.move_to(x + r, y)
    context.arc(x, y, r, 0, 2 * pi)
    context.set_source_rgba(1, 0, 1, self.alpha)
    context.fill_preserve()
    context.set_source_rgba(0, 0, 0, self.alpha)
    context.stroke()
    
    x2 = self.point_b.x
    y2 = self.point_b.y
    context.arc(x2, y2, r, 0, 2 * pi)
    context.set_source_rgba(1, 0, 1, self.alpha)
    context.fill_preserve()
    context.set_source_rgba(0, 0, 0, self.alpha)
    context.stroke()

    dr = sqrt((x2-x)**2 + (y2-y)**2)
    if dr > 0:
      context.move_to(x+r*(x2-x)/dr, y+r*(y2-y)/dr)
      context.line_to(x2-r*(x2-x)/dr, y2-r*(y2-y)/dr)
      context.stroke()

  def finished(self):
    return self.alpha <= 0.0

  def frame(self, time = None):
    """Advance one frame in animation"""
    if self.alpha <= 0.0: return False
    self.alpha -= 0.05
    return True

class TimeAnimation:
  def __init__(self, x, y):
    self.x = x
    self.y = y
    self.seconds = 0
    self.started = 0
    self.counting = False

  def draw(self, context, layout):
    layout.set_text("Time: %d" % self.seconds)
    layout.set_font_description(pango.FontDescription("Sans 16"))
    context.move_to(self.x, self.y)
    context.update_layout(layout)
    context.show_layout(layout)

  def finished(self):
    return False

  def frame(self, time):
    if self.counting:
      self.seconds = time - self.started

  def start(self):
    self.started = time.time()
    self.seconds = 0
    self.counting = True

  def stop(self):
    self.counting = False