#!/usr/bin/env python

# Copyright (C) 2007  Phyrum Tea <phyrum.tea@sign.ch>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import random

# TODO
class WordsearchReader:
  def read(self, data, filename):
    if data is None:
      data = Wordsearch()
    return read(data)

# TODO
class WordsearchWriter:

  def write(self, data, filename):
    file = open(filename, 'w')
    file.write(data.rows, data.cols)
    for row in data.fields:
      for col in row:
        file.write(col.char)
    
    for word in data.words:
      file.write(word.row)
      file.write(word.col)
      file.write(word.dir)
      file.write(word.status)
      file.write(word.word)
    file.write(data.author)
    file.close()
    
  def _writeProperties(self, data):
    """Write visual properties"""
    pass

  def _writeUserdata(self, data):
    """Write user data as guessed and revealed words. Total play time"""
    pass

class WordsearchCompiler:

  def __init__(self, rows, cols):
    self.rows = rows
    self.cols = cols
    self.wordsearch = Wordsearch(self.rows, self.cols)
    self.fields = self.wordsearch.fields

  def create(self, wordlist):
    self.wordlist = wordlist
    self.__initFields()
    self._fillRandom()
    #self.__displayFields()
    self._fillScan()
    self.__displayFields()
    self._fillFinish()
    self.__displayFields()
    self.wordsearch.words = self.used

    return self.wordsearch

  def __displayFields(self):
    for r in self.fields:
      for c in r:
        print c.char,
      print
    print

  def _fillRandom(self):
    """In phase 1 random words are placed randomly in the fields."""

    iMax = len(self.candidates)*10
    i = 0

    while i < iMax and len(self.candidates) > 0:
      d = random.randint(0, 3)
      w = random.randint(0, len(self.candidates)-1)
      word = self.candidates[w]
      l = len(word)
      if d == 0:
        d = Candidate.HORIZONTAL
        r = random.randint(0, self.rows-1)
        c = random.randint(0, self.cols-l-1) if l < self.cols else 0
      elif d == 1:
        d = Candidate.VERTICAL
        r = random.randint(0, self.rows-l-1) if l < self.rows else 0
        c = random.randint(0, self.cols-1) if l < self.cols else 0
      elif d == 2:
        d = Candidate.DOWNDIAGONAL
        r = random.randint(0, self.rows-l-1) if l < self.rows else 0
        c = random.randint(0, self.cols-l-1) if l < self.cols else 0
      elif d == 3:
        d = Candidate.UPDIAGONAL
        r = self.rows -1 - random.randint(0, self.rows-l-1) if l < self.rows else 0
        c = random.randint(0, self.cols-l-1) if l < self.cols else 0
      dr, dc = Candidate.steps[d]
      if self.__canset(word, r, c, dr, dc):
        self.__set(word, r, c, dr, dc)
        self.used.append(Candidate(word, r, c, d))
        self.candidates.remove(word)

      i += 1

  def _fillScan(self):
    'Choose remaining words randomly and try to fit it by brute force.'
    iMax = len(self.candidates)
    i = 0
    while i < iMax and len(self.candidates) > 0:
      w = i % len(self.candidates)
      word = self.candidates[w]
      r = random.randint(0, self.rows-1)
      set = False
      j = 0
      while j < self.rows and not(set):
        c = random.randint(0, self.cols-1)
        k = 0
        while k < self.cols and not(set):
          d = random.randint(0, 3)
          l = 0
          while l < 4 and not(set):
            if d == 0:
              e = Candidate.HORIZONTAL
            elif d == 1:
              e = Candidate.VERTICAL
            elif d == 2:
              e = Candidate.DOWNDIAGONAL
            elif d == 3:
              e = Candidate.UPDIAGONAL
            dr, dc = Candidate.steps[e]
            if self.__canset(word, r, c, dr, dc):
              self.__set(word, r, c, dr, dc)
              self.used.append(Candidate(word, r, c, e))
              self.candidates.remove(word)
              set = True
            d = (d+1) % 4
            l += 1
          c = (c+1) % self.cols
          k += 1
        r = (r+1) % self.rows
        j += 1
      i += 1

  def _fillFinish(self):
    'Each field without letter will get a randomly choosed letter.'
    
    for r in self.fields:
      for c in r:
        if c.char == ' ':
          c.char = unichr(random.randrange(ord('A'), ord('Z')))

  def __initFields(self):
    self.candidates = []
    self.used = []
    self.candidates.extend(self.wordlist)

  def __canset(self, word, row, col, dr, dc):
    r, c = row, col
    iMax = len(word)

    # test if word fits in fields 
    if dc > 0 and iMax + col >= self.cols: return False
    if dr > 0 and iMax + row >= self.rows: return False
    if dc < 0 and col - iMax < 0: return False
    if dr < 0 and row - iMax < 0: return False

    # test if word can be placed
    for i in xrange(iMax):
      if self.fields[r][c].char != ' ' and self.fields[r][c].char != word[i:i+1]:
        return False
      r += dr
      c += dc

    return True

  def __set(self, word, row, col, dr, dc):
    r, c = row, col
    for i in xrange(len(word)):
      self.fields[r][c].char = word[i:i+1]
      r += dr
      c += dc

class Wordsearch:
  def __init__(self, rows, cols):
    self.author = ''
    self.rows = rows
    self.cols = cols

    self.words = []
    fields = []
    for r in xrange(rows):
      fields.append([])
      for c in xrange(cols):
        fields[r].append(Field(r, c))

    self.fields = fields

  def get_guessed(self):
    """Return all candidates that are guessed by player."""
    return self.get_words(Candidate.STATUS_GUESSED)

  def get_hidden(self):
    """Return all candidates that are guessed by player."""
    return self.get_words(Candidate.STATUS_HIDDEN)

  def get_normal(self):
    return self.get_words(Candidate.STATUS_NORMAL)

  def get_revealed(self):
    """Return all candidates that are revelead to the player."""
    return self.get_words(Candidate.STATUS_REVEALED)

  def get_word(self, r1, c1, r2, c2):
    for w in self.words:
      l = len(w.word)
      if w.row == r1 and w.col == c1:
        dr, dc = Candidate.steps[w.dir]
        if dr == (r2 - r1 + 1)//l and dc == (c2 - c1 + 1)//l:
          return w
    return None

  def get_words(self, status):
    """Return all candidates that are revelead to the player."""
    data = []
    for w in self.words:
      if w.status == status:
        data.append(w)

    return data
    
class Candidate:
  HORIZONTAL = 1
  VERTICAL = 2
  DOWNDIAGONAL = 3
  UPDIAGONAL = 4
  R_HORIZONTAL = 5
  R_VERTICAL = 6
  R_DOWNDIAGONAL = 7
  R_UPDIAGONAL = 8

  steps = {HORIZONTAL : (0, 1)
         , VERTICAL : (1, 0)
         , DOWNDIAGONAL : (1, 1)
         , UPDIAGONAL : (-1, 1)
         , R_HORIZONTAL : (0, -1)
         , R_VERTICAL : (-1, 0)
         , R_DOWNDIAGONAL : (-1, -1)
         , R_UPDIAGONAL : (1, -1) }

  STATUS_NORMAL = 1
  STATUS_GUESSED = 2
  STATUS_REVEALED = 3
  STATUS_HIDDEN = 4

  def __init__(self, word, row, col, dir):
    self.word = word
    self.row = row
    self.col = col
    self.dir = dir
    self.status = Candidate.STATUS_NORMAL

class Field:
  def __init__(self, row = 0, col = 0, char = ' '):
    self.row = row
    self.col = col
    self.char = char
