# Copyright (C) 2007, Red Hat, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import os
import logging
from gettext import gettext as _
from gettext import dgettext as dgettext
import gconf
import pwd

import gtk
import gobject
import hippo

from sugar import env
from sugar.graphics import style
from sugar.graphics.icon import Icon
from sugar.graphics.entry import CanvasEntry
from sugar.graphics.xocolor import XoColor

from jarabe.intro import colorpicker
from jarabe.model.session import get_session_manager
import lang_model


_BACKGROUND_COLOR = style.COLOR_WHITE

_translate_language = lambda msg: dgettext('iso_639', msg)
_translate_country = lambda msg: dgettext('iso_3166', msg)


def create_profile(name, color=None):
    if not color:
        color = XoColor()

    client = gconf.client_get_default()
    client.set_string('/desktop/sugar/user/nick', name)
    client.set_string('/desktop/sugar/user/color', color.to_string())
    client.suggest_sync()

    # Generate keypair
    import commands
    keypath = os.path.join(env.get_profile_path(), 'owner.key')
    if not os.path.isfile(keypath):
        cmd = "ssh-keygen -q -t dsa -f %s -C '' -N ''" % keypath
        (s, o) = commands.getstatusoutput(cmd)
        if s != 0:
            logging.error('Could not generate key pair: %d %s', s, o)
    else:
        logging.error('Keypair exists, skip generation.')


class _Page(hippo.CanvasBox):
    __gproperties__ = {
        'valid': (bool, None, None, False, gobject.PARAM_READABLE),
    }

    def __init__(self, **kwargs):
        hippo.CanvasBox.__init__(self, **kwargs)
        self.valid = False

    def set_valid(self, valid):
        self.valid = valid
        self.notify('valid')

    def do_get_property(self, pspec):
        if pspec.name == 'valid':
            return self.valid

    def activate(self):
        pass


class _LanguagePage(_Page):
    def __init__(self, intro):
        _Page.__init__(self, xalign=hippo.ALIGNMENT_CENTER,
                       background_color=_BACKGROUND_COLOR.get_int(),
                       spacing=style.DEFAULT_SPACING,
                       orientation=hippo.ORIENTATION_HORIZONTAL,)

        self._intro = intro
        self._model = lang_model
        self.restart_alerts = list()
        self._lang_sid = 0
        self._selected_lang_count = 0
        self._labels = []
        self._stores = []
        self._comboboxes = []
        self._add_remove_boxes = []
        self._changed = False
        self._cursor_change_handler = None

        self._available_locales = self._model.read_all_languages()
        self._selected_locales = self._model.get_languages()

        vbox = gtk.VBox()
        vbox.set_size_request(600, 600)

        spacer = gtk.HBox()
        vbox.pack_start(spacer)

        explanation = _('Add languages in the order you prefer.'
                                      ' If a translation is not available,'
                                      ' the next in the list will be used.')

        self._text = gtk.Label(explanation)
        self._text.set_width_chars(100)
        self._text.set_line_wrap(True)
        self._text.set_alignment(0, 0)
        vbox.pack_start(self._text, False)

        scrolled = gtk.ScrolledWindow()
        viewport = gtk.Viewport()
        viewport.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("#ffffff"))
        scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        vbox.pack_start(scrolled, expand=True)

        self._table = gtk.Table(rows=1, columns=3, homogeneous=False)
        self._table.set_border_width(style.DEFAULT_SPACING * 2)
        viewport.add(self._table)
        scrolled.add(viewport)

        icon = Icon(icon_name="system-restart")
        self.apply_button = gtk.Button(label=_("Ok"))
        self.apply_button.set_image(icon)
        align = gtk.Alignment(0.5, 0.5, 0.5, 0.5)
        align.add(self.apply_button)
        self.apply_button.connect('clicked', self.__apply_button_clicked_cb)
        vbox.pack_start(align, expand=False, fill=False, padding=40)
        self.apply_button.props.sensitive=False

        vbox.show_all()

        box = hippo.CanvasWidget(widget=vbox)
        self.append(box)

        if gtk.widget_get_default_direction() == gtk.TEXT_DIR_RTL:
            self.reverse()

        self.set_valid(True)
        self.setup()

    def __apply_button_clicked_cb(self, button):
        selected_langs = self._get_selected_langs()
        self._model.set_languages_list(selected_langs)
        file(os.path.join(env.get_profile_path(), 'restart'), 'w').close()
        exit(0)

    def _add_row(self, locale_code=None):
        """Adds a row to the table"""

        self._selected_lang_count += 1

        self._table.resize(self._selected_lang_count, 3)

        label = gtk.Label(str=str(self._selected_lang_count))
        label.modify_fg(gtk.STATE_NORMAL,
            style.COLOR_SELECTION_GREY.get_gdk_color())
        self._labels.append(label)
        self._attach_to_table(label, 0, 1, padding=1)
        label.show()

        store = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING)
        for language, country, code in self._available_locales:
            description = '%s (%s)' % (_translate_language(language), \
                _translate_country(country))
            store.append([code, description])

        combobox = gtk.ComboBox(model=store)
        cell = gtk.CellRendererText()
        combobox.pack_start(cell)
        combobox.add_attribute(cell, 'text', 1)

        if locale_code:
            for row in store:
                lang = locale_code.split('.')[0]
                lang_column = row[0].split('.')[0]
                if lang in lang_column:
                    combobox.set_active_iter(row.iter)
                    break
        else:
            combobox.set_active(1)

        combobox.connect('changed', self.__combobox_changed_cb)

        self._stores.append(store)
        self._comboboxes.append(combobox)
        self._attach_to_table(combobox, 1, 2, yoptions=gtk.SHRINK)

        add_remove_box = self._create_add_remove_box()
        self._add_remove_boxes.append(add_remove_box)
        self._attach_to_table(add_remove_box, 2, 3)

        add_remove_box.show_all()

        if self._selected_lang_count > 1:
            previous_add_removes = self._add_remove_boxes[-2]
            previous_add_removes.hide_all()

        self._determine_add_remove_visibility()

        combobox.show()

    def _attach_to_table(self, widget, row, column, padding=20, \
            yoptions=gtk.FILL):
        self._table.attach(widget, row, column, \
            self._selected_lang_count - 1, self._selected_lang_count, \
            xoptions=gtk.FILL, yoptions=yoptions, xpadding=padding, \
                ypadding=padding)

    def _delete_last_row(self):
        """Deletes the last row of the table"""

        self._selected_lang_count -= 1

        label, add_remove_box, combobox, store_ = self._get_last_row()

        label.destroy()
        add_remove_box.destroy()
        combobox.destroy()

        self._table.resize(self._selected_lang_count, 3)

        self._add_remove_boxes[-1].show_all()

    def _get_last_row(self):
        label = self._labels.pop()
        add_remove_box = self._add_remove_boxes.pop()
        combobox = self._comboboxes.pop()
        store = self._stores.pop()

        return label, add_remove_box, combobox, store

    def setup(self):
        for locale in self._selected_locales:
            self._add_row(locale_code=locale)

    def undo(self):
        self._model.undo()
        self._lang_alert.hide()

    def _create_add_remove_box(self):
        """Creates gtk.Hbox with add/remove buttons"""
        add_icon = Icon(icon_name='list-add')

        add_button = gtk.Button()
        add_button.set_image(add_icon)
        add_button.connect('clicked',
                            self.__add_button_clicked_cb)

        remove_icon = Icon(icon_name='list-remove')
        remove_button = gtk.Button()
        remove_button.set_image(remove_icon)
        remove_button.connect('clicked',
                            self.__remove_button_clicked_cb)

        add_remove_box = gtk.HButtonBox()
        add_remove_box.set_layout(gtk.BUTTONBOX_START)
        add_remove_box.set_spacing(10)
        add_remove_box.pack_start(add_button)
        add_remove_box.pack_start(remove_button)

        return add_remove_box

    def __add_button_clicked_cb(self, button):
        self._add_row()
        self._check_change()

    def __remove_button_clicked_cb(self, button):
        self._delete_last_row()
        self._check_change()

    def __combobox_changed_cb(self, button):
        self._check_change()

    def _check_change(self):
        self.apply_button.props.sensitive=True
        self.set_valid(False)
        selected_langs = self._get_selected_langs()
        last_lang = selected_langs[-1]

        self._determine_add_remove_visibility(last_lang=last_lang)

        self._changed = (selected_langs != self._selected_locales)

        if self._changed == False:
            # The user reverted back to the original config
            self.set_valid(True)
            self.apply_button.props.sensitive=False
            return False
        else:
            self.set_valid(False)

    def _get_selected_langs(self):
        new_codes = []
        for combobox in self._comboboxes:
            it = combobox.get_active_iter()
            model = combobox.get_model()
            lang_code = model.get(it, 0)[0]
            new_codes.append(lang_code)

        return new_codes

    def _determine_add_remove_visibility(self, last_lang=None):
        # We should not let users add fallback languages for English (USA)
        # This is because the software is not usually _translated_ into English
        # which means that the fallback gets selected automatically

        if last_lang is None:
            selected_langs = self._get_selected_langs()
            last_lang = selected_langs[-1]

        add_remove_box = self._add_remove_boxes[-1]
        buttons = add_remove_box.get_children()
        add_button, remove_button = buttons

        if last_lang.startswith('en_US'):
            add_button.props.visible = False
        else:
            add_button.props.visible = True

        if self._selected_lang_count == 1:
            remove_button.props.visible = False
        else:
            remove_button.props.visible = True

    def __lang_timeout_cb(self, codes):
        self._lang_sid = 0
        self._model.set_languages_list(codes)
        self.restart_alerts.append('lang')
        self.needs_restart = True
        self._lang_alert.props.msg = self.restart_msg
        self._lang_alert.show()
        return False

    def activate(self):
        pass
        #self._entry.props.widget.grab_focus()


class _NamePage(_Page):
    def __init__(self, intro):
        _Page.__init__(self, xalign=hippo.ALIGNMENT_CENTER,
                       background_color=_BACKGROUND_COLOR.get_int(),
                       spacing=style.DEFAULT_SPACING,
                       orientation=hippo.ORIENTATION_HORIZONTAL,)

        self._intro = intro

        label = hippo.CanvasText(text=_('Name:'))
        self.append(label)

        self._entry = CanvasEntry(box_width=style.zoom(300))
        self._entry.set_background(_BACKGROUND_COLOR.get_html())
        self._entry.connect('notify::text', self._text_changed_cb)

        widget = self._entry.props.widget
        widget.set_max_length(45)

        self.append(self._entry)

        if gtk.widget_get_default_direction() == gtk.TEXT_DIR_RTL:
            self.reverse()

    def _text_changed_cb(self, entry, pspec):
        valid = len(entry.props.text.strip()) > 0
        self.set_valid(valid)

    def get_name(self):
        return self._entry.props.text

    def set_name(self, new_name):
        self._entry.props.text = new_name

    def activate(self):
        self._entry.props.widget.grab_focus()


class _ColorPage(_Page):
    def __init__(self, **kwargs):
        _Page.__init__(self, xalign=hippo.ALIGNMENT_CENTER,
                       background_color=_BACKGROUND_COLOR.get_int(),
                       spacing=style.DEFAULT_SPACING,
                       yalign=hippo.ALIGNMENT_CENTER, **kwargs)

        self._label = hippo.CanvasText(text=_('Click to change color:'),
                                       xalign=hippo.ALIGNMENT_CENTER)
        self.append(self._label)

        self._cp = colorpicker.ColorPicker(xalign=hippo.ALIGNMENT_CENTER)
        self.append(self._cp)

        self._color = self._cp.get_color()
        self.set_valid(True)

    def get_color(self):
        return self._cp.get_color()


class _IntroBox(hippo.CanvasBox):
    __gsignals__ = {
        'done': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                 ([gobject.TYPE_PYOBJECT, gobject.TYPE_PYOBJECT])),
    }

    PAGE_LANGUAGE = 0
    PAGE_NAME = 1
    PAGE_COLOR = 2

    PAGE_FIRST = PAGE_LANGUAGE
    PAGE_LAST = PAGE_COLOR

    def __init__(self):
        hippo.CanvasBox.__init__(self, padding=style.zoom(30),
                                 background_color=_BACKGROUND_COLOR.get_int())

        self._page = self.PAGE_LANGUAGE
        self._language_page = _LanguagePage(self)
        self._name_page = _NamePage(self)
        self._color_page = _ColorPage()
        self._current_page = None
        self._next_button = None

        client = gconf.client_get_default()
        default_nick = client.get_string('/desktop/sugar/user/default_nick')
        if default_nick != 'disabled':
            self._page = self.PAGE_COLOR
            if default_nick == 'system':
                pwd_entry = pwd.getpwuid(os.getuid())
                default_nick = (pwd_entry.pw_gecos.split(',')[0] or
                                pwd_entry.pw_name)
            self._name_page.set_name(default_nick)

        self._setup_page()

    def _setup_page(self):
        self.remove_all()

        if self._page == self.PAGE_LANGUAGE:
            self._current_page = self._language_page
        elif self._page == self.PAGE_NAME:
            self._current_page = self._name_page
        elif self._page == self.PAGE_COLOR:
            self._current_page = self._color_page

        self.append(self._current_page, hippo.PACK_EXPAND)

        button_box = hippo.CanvasBox(orientation=hippo.ORIENTATION_HORIZONTAL)

        if self._page != self.PAGE_FIRST:
            back_button = hippo.CanvasButton(text=_('Back'))
            image = Icon(icon_name='go-left')
            back_button.props.widget.set_image(image)
            back_button.connect('activated', self._back_activated_cb)
            button_box.append(back_button)

        spacer = hippo.CanvasBox()
        button_box.append(spacer, hippo.PACK_EXPAND)

        self._next_button = hippo.CanvasButton()
        image = Icon(icon_name='go-right')
        self._next_button.props.widget.set_image(image)

        if self._page == self.PAGE_LAST:
            self._next_button.props.text = _('Done')
            self._next_button.connect('activated', self._done_activated_cb)
        else:
            self._next_button.props.text = _('Next')
            self._next_button.connect('activated', self._next_activated_cb)

        self._current_page.activate()

        self._update_next_button()
        button_box.append(self._next_button)

        self._current_page.connect('notify::valid',
                                   self._page_valid_changed_cb)
        self.append(button_box)

        if gtk.widget_get_default_direction() == gtk.TEXT_DIR_RTL:
            button_box.reverse()

    def _update_next_button(self):
        widget = self._next_button.props.widget
        widget.props.sensitive = self._current_page.props.valid

    def _page_valid_changed_cb(self, page, pspec):
        self._update_next_button()

    def _back_activated_cb(self, item):
        self.back()

    def back(self):
        if self._page != self.PAGE_FIRST:
            self._page -= 1
            self._setup_page()

    def _next_activated_cb(self, item):
        self.next()

    def next(self):
        if self._page == self.PAGE_LAST:
            self.done()
        if self._current_page.props.valid:
            self._page += 1
            self._setup_page()

    def _done_activated_cb(self, item):
        self.done()

    def done(self):
        name = self._name_page.get_name()
        color = self._color_page.get_color()

        self.emit('done', name, color)


class IntroWindow(gtk.Window):
    def __init__(self):
        gtk.Window.__init__(self)

        self.props.decorated = False
        self.maximize()

        self._canvas = hippo.Canvas()
        self._intro_box = _IntroBox()
        self._intro_box.connect('done', self._done_cb)
        self._canvas.set_root(self._intro_box)

        self.add(self._canvas)
        self._canvas.show()
        self.connect('key-press-event', self.__key_press_cb)

    def _done_cb(self, box, name, color):
        self.hide()
        gobject.idle_add(self._create_profile_cb, name, color)

    def _create_profile_cb(self, name, color):
        create_profile(name, color)
        gtk.main_quit()

        return False

    def __key_press_cb(self, widget, event):
        if gtk.gdk.keyval_name(event.keyval) == 'Return':
            self._intro_box.next()
            return True
        elif gtk.gdk.keyval_name(event.keyval) == 'Escape':
            self._intro_box.back()
            return True
        return False


if __name__ == '__main__':
    w = IntroWindow()
    w.show()
    w.connect('destroy', gtk.main_quit)
    gtk.main()
