/*
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * PRECISION INSIGHT AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */

#include "drmP.h"
#include "drm.h"
#include "smi_drm.h"
#include "smi_drv.h"

static int smi_driver_load(struct drm_device *dev, unsigned long chipset);
static int smi_driver_firstopen(struct drm_device *dev);
static void smi_driver_lastclose(struct drm_device *dev);
static int smi_driver_unload(struct drm_device *dev);
static irqreturn_t smi_driver_irq_handler(DRM_IRQ_ARGS);
static void smi_driver_irq_preinstall(struct drm_device *dev);
/* The return type is void for 2.6.24 */
static int smi_driver_irq_postinstall(struct drm_device *dev);
static void smi_driver_irq_uninstall(struct drm_device *dev);
static int smi_driver_pci_probe(struct pci_dev *pdev,
				const struct pci_device_id *ent);

#ifdef DEBUG
int smi_indent;
#endif

static int64_t test[] = {
	((smi_cli_load_reg << 28) | 0x100014) | (0xff00LL << 32),
	((smi_cli_load_reg << 28) | 0x100034) | (0xffffffffLL << 32),
	((smi_cli_load_reg << 28) | 0x100038) | (0xffffffffLL << 32),
	((smi_cli_load_reg << 28) | 0x100004) | (0LL << 32),
	((smi_cli_load_reg << 28) | 0x100008) | (((1024LL << 16) | 600LL) << 32),
	((smi_cli_load_reg << 28) | 0x100048) | (127 << 32),
	((smi_cli_load_reg << 28) | 0x10000c) | (0x80010000LL << 32),
	/* wait for idle 2d engine */
	((smi_cli_status_test << 28) | (1LL << 32)),    
	((smi_cli_finish << 28) | 1)
};

static struct pci_device_id pciidlist[] = {
	{0x126f, 0x0501, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0}
};

static int smi_cli_verify(struct drm_device *dev, int32_t offset, int reloc);
static int smi_ioctl_batch(struct drm_device *dev, void *data,
			   struct drm_file *file_priv);
static int smi_ioctl_flush(struct drm_device *dev, void *data,
			   struct drm_file *file_priv);

struct drm_ioctl_desc smi_ioctls[] = {
	DRM_IOCTL_DEF(DRM_SMI_BATCH, smi_ioctl_batch,
		      DRM_AUTH | DRM_MASTER | DRM_ROOT_ONLY),
	DRM_IOCTL_DEF(DRM_SMI_FLUSH, smi_ioctl_flush,
		      DRM_AUTH | DRM_MASTER | DRM_ROOT_ONLY),
};

static struct drm_driver smi_driver = {
	.load = smi_driver_load,
	.firstopen = smi_driver_firstopen,
	.lastclose = smi_driver_lastclose,
	.unload = smi_driver_unload,
	.ioctls = smi_ioctls,
	.num_ioctls = DRM_ARRAY_SIZE(smi_ioctls),
	.irq_preinstall = smi_driver_irq_preinstall,
	.irq_postinstall = smi_driver_irq_postinstall,
	.irq_uninstall = smi_driver_irq_uninstall,
	.irq_handler = smi_driver_irq_handler,
	.get_map_ofs = drm_core_get_map_ofs,
	.get_reg_ofs = drm_core_get_reg_ofs,
	.fops = {
		.owner = THIS_MODULE,
		.open = drm_open,
		.release = drm_release,
		.ioctl = drm_ioctl,
		.mmap = drm_mmap,
		.poll = drm_poll,
		.fasync = drm_fasync,
	},
	.pci_driver = {
		.name = DRIVER_NAME,
		.id_table = pciidlist,
		.probe = smi_driver_pci_probe,
#if defined(__linux__) && (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,24))
		.remove = __devexit_p(drm_cleanup_pci),
#endif
	},
	.name = DRIVER_NAME,
	.desc = DRIVER_DESC,
	.date = DRIVER_DATE,
	.major = DRIVER_MAJOR,
	.minor = DRIVER_MINOR,
	.patchlevel = DRIVER_PATCHLEVEL,
	.driver_features = DRIVER_HAVE_IRQ | DRIVER_IRQ_SHARED,
};

static int smi_driver_load(struct drm_device *dev, unsigned long chipset)
{
	drm_smi_private_t *smi_dev;

	ENTER();
	smi_dev = drm_calloc(1, sizeof(drm_smi_private_t), DRM_MEM_DRIVER);
	if (smi_dev == NULL)
		LEAVE(-ENOMEM);

	dev->dev_private = (void *)smi_dev;
	smi_dev->chipset = dev->pci_device;

	LEAVE(0);
}

static int smi_driver_firstopen(struct drm_device *dev)
{
	drm_smi_private_t *smi_dev = dev->dev_private;
	int mmio_resource, mmio_offset, mmio_size;
	int batch_size;
	int ret;

	ENTER();
	printk("** smi_dev->chipset = %x\n", smi_dev->chipset);
	/* chipset specific values */
	switch (smi_dev->chipset) {
	case 0x501:
		mmio_resource = 1;
		mmio_offset = 0;
		mmio_size = 0x200000;
		batch_size = 32768;
		break;
	default:
		mmio_resource = 0;
		mmio_offset = 0x400000;
		mmio_size = 0x10000;
		batch_size = 32768;
		break;
	}

	{
		int i;

		for (i = 0; i < 4; i++)
			printk("res: %d = 0x%08x - 0x%08x\n",
			       i,
			       drm_get_resource_start(dev, i),
			       drm_get_resource_len(dev, i)
			       );
	}

	/* map mmio registers */
	ret = drm_addmap(dev, drm_get_resource_start(dev, mmio_resource)
			 + mmio_offset, mmio_size, _DRM_REGISTERS, 0,
			 &smi_dev->mmio);
	if (ret)
		LEAVE(ret);
	printk("mmio offset/size = 0x08%x/0x%08x\n",
	       smi_dev->mmio->offset,
	       smi_dev->mmio->size);
	printk("mmio type/flags = 0x08%x/0x%08x\n",
	       smi_dev->mmio->type,
	       smi_dev->mmio->flags);
	printk("mmio handle/mtrr = 0x%p/%d\n",
	       smi_dev->mmio->handle,
	       smi_dev->mmio->mtrr);

	/* map batch buffer */
	/* FIXME this requires drm_addmap() knowing that if offset is
	 * not 0, it should use it as maxaddr parameter to drm_pci_alloc(),
	 * otherwise it will default to the highest possible 32 bits ix86
	 * value.
	 * For the smi501/502, the command list buffer must reside below
	 * 128Mb, and dma buffers must reside below 64Mb.
	 *   DMA is really restricted to a 64Mb window, configurable
	 * in mmio 0x58 - PCI Master Base Address.
	 */
	ret = drm_addmap(dev, 0x8000000, (batch_size + ~PAGE_MASK) & PAGE_MASK,
			 _DRM_CONSISTENT, 0, &smi_dev->batch);

	if (ret)
		LEAVE(ret);

	DRM_INIT_WAITQUEUE(&smi_dev->cli_wait);

	{
		int		i;

		printk("SMI501 System Setup:\n");
		for (i = 0x00; i <= 0x74; i += 4)
			printk("\t%08x: %08x\n", i, SMI_READ(i));

		printk("SMI501 Display Setup:\n");
		for (i = 0x80000; i <= 0x80400; i += 4)
			printk("\t%08x: %08x\n", i, SMI_READ(i));
	}

#if 0
	{
		/* This does, by writting to mmio registers, what the test
		 * command list code should be doing ... */
		SMI_WRITE(0x100014, 0x000000ff);
		SMI_WRITE(0x100034, 0xffffffff);
		SMI_WRITE(0x100038, 0xffffffff);
		SMI_WRITE(0x100004, 0x00000000);
		SMI_WRITE(0x100008, (1024 << 16) | 600);
		SMI_WRITE(0x100048, 127);
		SMI_WRITE(0x10000c, 0x80040000);
	}
#endif

	{
		int i;
		drm_smi_cli_addr_t cli;

		for (i = 0; i < sizeof(test) / sizeof(int64_t); i++)
			printk("-> %016llx\n", test[i]);

		printk("handle : 0x%08x\n", smi_dev->batch->handle);
		printk("offset : 0x%08x\n", smi_dev->batch->offset);
		printk("size   : 0x%08x\n", smi_dev->batch->size);

		printk("prev : 0x%08x\n", SMI_READ(SMI_CLI_ADDRESS));

		cli.value = SMI_READ(SMI_CLI_ADDRESS);
		cli.f.start = 0;
		SMI_WRITE(SMI_CLI_ADDRESS, cli.value);

		printk("stop : 0x%08x\n", SMI_READ(SMI_CLI_ADDRESS));

		memcpy(smi_dev->batch->handle, test, sizeof(test));
		cli.f.address = smi_dev->batch->offset;

		SMI_WRITE(SMI_CLI_ADDRESS, cli.value);
		printk("new  : 0x%08x\n", SMI_READ(SMI_CLI_ADDRESS));

		cli.f.start = 1;
		SMI_WRITE(SMI_CLI_ADDRESS, cli.value);
		printk("start: 0x%08x\n", SMI_READ(SMI_CLI_ADDRESS));
	}

	LEAVE(ret);
}

static void smi_driver_lastclose(struct drm_device *dev)
{
	drm_smi_private_t *smi_dev = dev->dev_private;

	ENTER();
	if (!smi_dev)
		LEAVE();

	if (smi_dev->mmio) {
		drm_rmmap(dev, smi_dev->mmio);
		smi_dev->mmio = NULL;
	}

	if (smi_dev->batch) {
		drm_rmmap(dev, smi_dev->batch);
		smi_dev->batch = NULL;
	}

	LEAVE();
}

static int smi_driver_unload(struct drm_device *dev)
{
	drm_smi_private_t *smi_dev = dev->dev_private;

	ENTER();
	drm_free(smi_dev, sizeof(*smi_dev), DRM_MEM_DRIVER);

	LEAVE(0);
}

static int smi_cli_verify(struct drm_device *dev, int32_t offset, int reloc)
{
	drm_smi_private_t *smi_dev = dev->dev_private;
	smi_cli_entry_t *entry;
	int64_t *data;
	void *mem, *reg;
	int32_t size;

	ENTER();

	data = smi_dev->batch->handle + offset / sizeof(int64_t);
	size = smi_dev->batch->size / sizeof(int64_t);

	for (; offset < size; data++) {
		entry = (smi_cli_entry_t *)data;
		switch ((cli_cmd_code_t)(entry->f.cmd & 0xf)) {
		case smi_cli_load_mem:
			/* mem must be 64 bits aligned */
			mem = virt_to_phys((void *)(long)(entry->f.base << 2));
			if ((long)mem & 0xfffffffff000000fLL)
				LEAVE(1);
			if (reloc)
				entry->f.base = ((long)mem) >> 2;
			break;
		case smi_cli_load_reg:
			/* reg must be 32 bits aligned in range 0 - 0x1fffff */
			reg = virt_to_phys((void *)(long)(entry->f.base << 2));
			if ((long)reg & 0xffffffffffe00003LL)
				LEAVE(1);
			if (reloc)
				entry->f.base = ((long)reg) >> 2;
			break;
		case smi_cli_load_mem_imm:
			/* mem must be 32 bits aligned */
			mem = virt_to_phys((void *)(long)(entry->f.base << 2));
			if ((long)mem & 0xfffffffff0000003LL)
				LEAVE(1);
			if (reloc)
				entry->f.base = ((long)mem) >> 2;
			/* data is number of 32 bit words to copy, with
			 * possible pad to align at 64 bits. */
			data += (entry->f.data + 1) >> 1;
			break;
		case smi_cli_load_reg_imm:
			/* reg must be 32 bits aligned in range 0 - 0x1fffff */
			reg = virt_to_phys((void *)(long)(entry->f.base << 2));
			if ((long)reg & 0xffffffffffe00003LL)
				LEAVE(1);
			if (reloc)
				entry->f.base = ((long)reg) >> 2;
			/* data is number of 32 bit words to copy, with
			 * possible pad to align at 64 bits. */
			data += (entry->f.data + 1) >> 1;
			break;
		case smi_cli_load_mem_ind:
			/* mem must be 32 bits aligned */
			mem = virt_to_phys((void *)(long)(entry->f.base << 2));
			if ((long)mem & 0xfffffffff0000003LL)
				LEAVE(1);
			if (reloc)
				entry->f.base = ((long)mem) >> 2;
			++data;
			/* source data */
			mem = virt_to_phys((void *)(long)(entry->f.base << 2));
			if ((long)mem & 0xfffffffff0000003LL)
				LEAVE(1);
			if (reloc)
				entry->f.base = ((long)mem) >> 2;
			break;
		case smi_cli_load_reg_ind:
			/* reg must be 32 bits aligned in range 0 - 0x1fffff */
			reg = virt_to_phys((void *)(long)(entry->f.base << 2));
			if ((long)reg & 0xffffffffffe00003LL)
				LEAVE(1);
			if (reloc)
				entry->f.base = ((long)reg) >> 2;
			++data;
			/* source data */
			mem = virt_to_phys((void *)(long)(entry->f.base << 2));
			if ((long)mem & 0xfffffffff0000003LL)
				LEAVE(1);
			if (reloc)
				entry->f.base = ((long)mem) >> 2;
			break;
		case smi_cli_status_test:
			break;
		case smi_cli_finish:
			LEAVE(0);
		case smi_cli_goto:
			/* If a relative goto */
			if (entry->f.data & 1) {
				/* FIXME value is 32 or 64 bits? */
				mem = data + ((entry->value & 0xffff) >> 3);
				/* Only allow goto inside command buffer */
				if (mem < smi_dev->batch->handle
				    || mem >= smi_dev->batch->handle + size)
					LEAVE(1);
				/* Follow jump code */
				if (smi_cli_verify(dev,
						   (mem
						    - smi_dev->batch->handle)
						   / sizeof(int64_t), 0))
					LEAVE(1);
			}
			else
				/* Goto immediate offset not supported */
				LEAVE(1);
			break;
		case smi_cli_gosub:
			/* Subroutines not supported */
			LEAVE(1);
		case smi_cli_return:
			/* Subroutines not supported */
			LEAVE(1);
		case smi_cli_cond_jump:
			mem = data + ((entry->value & 0xffff) >> 3);
			/* Only allow jump inside command buffer */
			if (mem < smi_dev->batch->handle
			    || mem >= smi_dev->batch->handle + size)
				LEAVE(1);
			/* Follow jump code */
			if (smi_cli_verify(dev, (mem - smi_dev->batch->handle)
					   / sizeof(int64_t), 0))
				LEAVE(1);
			break;
		default:
			LEAVE(1);
		}
	}
	LEAVE(1);
}

static int smi_ioctl_batch(struct drm_device *dev, void *data,
			   struct drm_file *file_priv)
{
	drm_smi_private_t *smi_dev = dev->dev_private;
	drm_smi_batch_t	*batch = data;
	drm_smi_cli_addr_t cli;
	int ret = 0;

	ENTER();
	if (batch->offset + batch->length >= smi_dev->batch->size)
		LEAVE(-ENOMEM);
	/* Commands must be 64 bit aligned. */
	if (batch->offset & 7)
		LEAVE(-EINVAL);

	/* Ensure the cli is idle before overwriting it's buffer. */
	DRM_WAIT_ON(ret, smi_dev->cli_wait, 3 * DRM_HZ,
		    (cli.value = SMI_READ(SMI_CLI_ADDRESS), cli.f.idle));
	if (ret)
		LEAVE(ret);

	if (DRM_COPY_FROM_USER(smi_dev->batch->handle + batch->offset,
			       batch->buffer, batch->length))
		LEAVE(-EFAULT);

	if (smi_cli_verify(dev, batch->offset, 1))
		LEAVE(-EINVAL);


	cli.f.address	= smi_dev->batch->offset;
	cli.f.start	= 1;
	SMI_WRITE(SMI_CLI_ADDRESS, cli.value);

	LEAVE(ret);
}

static int smi_ioctl_flush(struct drm_device *dev, void *data,
			   struct drm_file *file_priv)
{
	drm_smi_private_t *smi_dev = dev->dev_private;
	drm_smi_cli_addr_t cli;
	int ret = 0;

	ENTER();

	DRM_WAIT_ON(ret, smi_dev->cli_wait, 3 * DRM_HZ,
		    (cli.value = SMI_READ(SMI_CLI_ADDRESS), cli.f.idle));

	LEAVE(ret);
}

static void smi_driver_irq_preinstall(struct drm_device *dev)
{
	ENTER();
	LEAVE();
}

static int smi_driver_irq_postinstall(struct drm_device *dev)
{
	drm_smi_private_t *smi_dev = dev->dev_private;
	drm_smi_irq_t irq;

	ENTER();
	switch (smi_dev->chipset) {
	case 0x501:
		/* enable command list interpreter interrupt */
		irq.value = SMI_READ(SMI_IRQ_MASK);
		irq.f.cli = 1;
		SMI_WRITE(SMI_IRQ_MASK, irq.value);
		break;
	}
	LEAVE(0);
}

static void smi_driver_irq_uninstall(struct drm_device *dev)
{
	drm_smi_private_t *smi_dev = dev->dev_private;
	drm_smi_irq_t irq;

	ENTER();
	switch (smi_dev->chipset) {
	case 0x501:
		/* disable command list interpreter interrupt */
		irq.value = SMI_READ(SMI_IRQ_MASK);
		irq.f.cli = 0;
		SMI_WRITE(SMI_IRQ_MASK, irq.value);
		break;
	}
	LEAVE();
}

static irqreturn_t smi_driver_irq_handler(DRM_IRQ_ARGS)
{
	struct drm_device *dev = (struct drm_device *)arg;
	drm_smi_private_t *smi_dev = (drm_smi_private_t *)dev->dev_private;
	irqreturn_t ret = IRQ_NONE;
	drm_smi_irq_t smi_irq;

	printk("irq ... ");
	switch (smi_dev->chipset) {
	case 0x501:
		smi_irq.value = SMI_READ(SMI_IRQ_STATUS);
		/* ACK, a FINISH command have been processed */
		if (smi_irq.f.cli) {
			DRM_WAKEUP(&smi_dev->cli_wait);
			ret = IRQ_HANDLED;
		}
		break;
	}
	printk("%s\n", ret == IRQ_HANDLED ? "handled" : "none");

	return ret;
}

static int smi_driver_pci_probe(struct pci_dev *pdev,
				const struct pci_device_id *ent)
{
	ENTER();
	LEAVE(drm_get_dev(pdev, ent, &smi_driver));
}

static int __init smi_init(void)
{
	ENTER();
#if defined(__linux__) && (LINUX_VERSION_CODE > KERNEL_VERSION(2,6,24))
	LEAVE(drm_init(&smi_driver, pciidlist));
#else
	LEAVE(drm_init(&smi_driver));
#endif
}

static void __exit smi_exit(void)
{
	ENTER();
	drm_exit(&smi_driver);
	LEAVE();
}

module_init(smi_init);
module_exit(smi_exit);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL and a BSD license stamp");
