/*
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * PRECISION INSIGHT AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 */

#ifndef _SMI_DRM_H_
#define _SMI_DRM_H_

#include <linux/types.h>

typedef enum cli_cmd_code {
	/*	Load Memory 0000b
	 *
	 *	0:0	When this bit is programmed to "0", the 32-bit DWord
	 *		data (bits [63:32]) is written to the Memory Address.
	 *		When this bit is programmed to "1", the 16-bit Word
	 *		data (bits [47:32]) is written to the Memory Address.
	 *	1:27	The Memory Address to write data to. Bits [3:0] are
	 *		hardwired to "0" since all Memory Addresses should be
	 *		128-bit aligned.
	 *	28:31	0000b
	 *	32:61	The data to be loaded in the memory address specified
	 *		by Memory Address. The data format is either 32-bit
	 *		DWords or 16-bit Words.
	 *	62:63	Bits [63:62] are the byte-enable signals for the Word
	 *		data. They are active high.
	 */
	smi_cli_load_mem,

	/*	Load Register 0001b
	 *
	 *	0:27	The register address (in the space 0x00000000 -
	 *		0x001FFFFF) to write data to. Bits [0:1] are
	 *		hardwired to "0"since all register addresses should
	 *		be 32-bit aligned.
	 *	28:31	0001b
	 *	32:63	The data to be loaded in the register specified by
	 *		Register Address.
	 */
	smi_cli_load_reg,

	/*	Load Memory Immediate 0010b
	 *
	 *	0:27	The starting memory address to write data to.
	 *		Bits [1:0] are hardwired to "0".
	 *	28:31	0010b
	 *	32:63	The number of DWORDs to load into the memory.
	 *
	 *   The data that must be loaded into the memory directly follows
	 * this command. Make sure the correct number of DWORDs (DWORD Count)
	 * is provided, otherwise unpredicted results will happen. Also, if
	 * an odd number of DWORDs is specified, the last DWORD should be
	 * padded with a dummy DWORD to align the next command to 64-bit again.
	 */
	smi_cli_load_mem_imm,

	/*	Load Register Immediate 0011b
	 *
	 *	0:27	The register address (in the space 0x00000000 -
	 *		0x001FFFFF) to write data to. Bits [0:1] are
	 *		hardwired to "0"since all register addresses should
	 *		be 32-bit aligned.
	 *	28:31	0011b
	 *	32:63	The number of DWORDs to load into the registers.
	 *
	 *   The data that must be loaded into the registers directly follows
	 * this command. Make sure the correct number of DWORDs (DWORD Count)
	 * is provided, otherwise unpredicted results will happen. Also, if
	 * an odd number of DWORDs is specified, the last DWORD should be
	 * padded with a dummy DWORD to align the next command to 64-bit again.
	 */
	smi_cli_load_reg_imm,

	/*	Load Memory Indirect 0100b
	 *
	 *	0:27	The starting memory address to write data to.
	 *		Bits [1:0] are hardwired to "0".
	 *	28:31	0100b
	 *	32:63	The number of DWORDs to copy into the memory.
	 *	64:91	The starting memory address to read data from.
	 *		Bits [65:64] are hardwired to "0".
	 *	92:127	Unused.
	 *
	 *   This command copies data from the memory location specified
	 * by Source Memory Address into the memory location specified by
	 * Memory Address. The DWORD Count specifies the number of DWORDs
	 * to copy. This command is most useful to copy texture, bitmap,
	 * or vertex data to off-screen memory for caching purposes.
	 */
	smi_cli_load_mem_ind,

	/*	Load Register Indirect 0101b
	 *
	 *	0:27	The register address (in the space 0x00000000 -
	 *		0x001FFFFF) to write data to. Bits [0:1] are
	 *		hardwired to "0"since all register addresses should
	 *		be 32-bit aligned.
	 *	28:31	0101b
	 *	32:63	The number of DWORDs to copy into the memory.
	 *	64:91	The starting memory address to read data from.
	 *		Bits [65:64] are hardwired to "0".
	 *	92:127	Unused.
	 *
	 *   This command copies data from the memory location specified
	 * by Source Memory Address into the register bank location specified
	 * by Register Address. The DWORD Count specifies the number of DWORDs
	 * to copy. This command is most useful to copy texture, bitmap,
	 * or vertex data to the engine FIFOs for processing.
	 */
	smi_cli_load_reg_ind,

	/*	Status Test 0110b
	 *
	 *	0:0	2D Drawing Engine
	 *		(0 = idle, 1 = busy).
	 *	1:1	2D and Color Space Conversion command FIFO
	 *		(0 = not empty, 1 = empty).
	 *	2:2	2D Setup Engine (0 = idle, 1 = busy).
	 *	3:10	Unused.
	 *	11:11	Vertical Sync for Panel pipe
	 *		(0 = not active, 1 = active).
	 *	12:12	Vertical Sync for CRT pipe
	 *		(0 = not active, 1 = active).
	 *	13:13	Panel Graphics Layer status bit.
	 *	14:14	Video Layer status bit.
	 *	15:15	Current Video Layer field for BOB
	 *		(0 = odd, 1 = even).
	 *	16:16	CRT Graphics Layer status bit.
	 *	17:17	Memory DMA busy bit.
	 *	18:18	Color Space Conversion busy bit.
	 *	19:19	Command FIFO on HIF bus
	 *		(0 = not empty, 1 = empty).
	 *	20:20	2D and Color Space Conversion memory FIFO
	 *		(0 = not empty, 1 = empty).
	 *	21:27	Unused.
	 *	28:31	0110b
	 *	32:52	Bits Values
	 *	53:63	Unused
	 *
	 *   The Status Test command will wait until the requested status
	 * is met. The value of the Status Test register is masked with the
	 * internal hardware state and compared to the state in the Bit Values.
	 * If the result does not equal the Bit Values, the command list
	 * interpreter will wait until the hardware status changes. The
	 * pseudo code looks like this:
	 *	WHILE (Hardware State & Mask [20:0]
	 *	       != Bit Values [52:32] & Mask [20:0]) NOP;
	 */
	smi_cli_status_test,

	/*	Finish 1000b
	 *
	 *	0:0	If the Interrupt bit is set, the FINISH command
	 *		will generate an interrupt that can still be
	 *		masked by the Command List mask bit in the Interrupt
	 *		Mask register. When an interrupt is generated, the
	 *		Command List bit in Interrupt Status register will
	 *		be set to "1".
	 *	1:27	Unused
	 *	28:31	1000b
	 *	32:63	Unused
	 *
	 *   The FINISH command stops executing commands in the Command List
	 * and clears the Start bit ([31]) of the Command List Address register.
	 */
	smi_cli_finish = 8,

	/*	Goto 1001b
	 *
	 *	0:27	The address of the new code to execute.
	 *		Bits [2:0] are hardwired to "0" since all addresses
	 *		need to be 64-bit aligned.
	 *	28:31	1001b
	 *	32:32	If the Relative bit is set, the specified Address
	 *		is relative to the address of the current command
	 *		(signed addition).
	 *	33:63	Unused.
	 *
	 *   The GOTO command will jump to the Command List code located at
	 * the specified Address.
	 */
	smi_cli_goto,

	/*	Gosub 1011b
	 *
	 *	0:27	The address of the new code to execute.
	 *		Bits [2:0] are hardwired to "0" since all addresses
	 *		need to be 64-bit aligned.
	 *	28:31	1011b
	 *	32:32	If the Relative bit is set, the specified Address
	 *		is relative to the address of the current command
	 *		(signed addition).
	 *	33:63	Unused.
	 *
	 *   The GOSUB command will store the address of the next instruction
	 * it would execute in the Command List Return Address register and
	 * starts executing the Command List code located at the specified
	 * Address.
	 */
	smi_cli_gosub,

	/*	Return 1010b
	 *
	 *	0:27	Unused
	 *	28:31	1011b
	 *	32:63	Unused.
	 *
	 *   The RETURN command will jump to the address specified in the
	 * Command List Return Address register. The RETURN command should
	 * terminate a subroutine that is being called by GOSUB.
	 */
	smi_cli_return,

	/*	Conditional Jump 1100b
	 *
	 *	0:27	A signed relative value that will be added to the
	 *		address of the next command in the Command List if
	 *		the result of the condition is TRUE. Bits [2:0] are
	 *		hardwired to "0" since all addresses need to be 64-bit
	 *		aligned.
	 *	28:31	1100b
	 *	32:63	The Condition field consists of a 32-bit mask that
	 *		will be applied to the Command List Condition Register.
	 *		If the result of this mask is TRUE (any bit set), the
	 *		condition shall return TRUE and the jump is taken by
	 *		adding the signed value of Address to the address of
	 *		the next command in the Command List.
	 *		The formula of the condition is:
	 *		RESULT = Condition * Command List Condition register
	 */
	smi_cli_cond_jump
} cli_cmd_code_t;

#define	bits(lo, hi)			hi + 1 - lo

/* Generic command list entry that matches most commands pattern */
typedef union smi_cli_entry {
    struct {
	int64_t	align	: bits( 0,  1);
	int64_t	base	: bits( 2, 27);
	int64_t	cmd	: bits(28, 31);
	int64_t	data	: bits(32, 63);
	
    } f;
    int64_t	value;
} smi_cli_entry_t;

typedef struct drm_smi_batch {
	int64_t	*buffer;
	int32_t	 length;
	int32_t	 offset;
} drm_smi_batch_t;

#define DRM_SMI_BATCH				0x00
#define DRM_SMI_FLUSH				0x01

#define DRM_IOCTL_SMI_BATCH						\
	DRM_IOW(DRM_COMMAND_BASE + DRM_SMI_BATCH, drm_smi_batch_t)
#define DRM_IOCTL_SMI_FLUSH						\
	DRM_IO(DRM_COMMAND_BASE + DRM_SMI_FLUSH)

#endif /* _SMI_DRM_H_ */
