# Copyright (C) 2009 Paraguay Educa, Martin Abente
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  US

import gconf
import gtk
import os
import locale
import logging

from xml.etree.cElementTree import ElementTree
from gettext import gettext as _

from jarabe.model.network import GSM_USERNAME_PATH, GSM_PASSWORD_PATH, \
                                 GSM_NUMBER_PATH, GSM_APN_PATH, GSM_PIN_PATH, \
                                 GSM_PUK_PATH

from cpsection.modemconfiguration.config import PROVIDERS_PATH, \
                                                PROVIDERS_FORMAT_SUPPORTED, \
                                                COUNTRY_CODES_PATH


def get_username():
    client = gconf.client_get_default()
    return client.get_string(GSM_USERNAME_PATH) or ''


def get_password():
    client = gconf.client_get_default()
    return client.get_string(GSM_PASSWORD_PATH) or ''


def get_number():
    client = gconf.client_get_default()
    return client.get_string(GSM_NUMBER_PATH) or ''


def get_apn():
    client = gconf.client_get_default()
    return client.get_string(GSM_APN_PATH) or ''


def get_pin():
    client = gconf.client_get_default()
    return client.get_string(GSM_PIN_PATH) or ''


def get_puk():
    client = gconf.client_get_default()
    return client.get_string(GSM_PUK_PATH) or ''


def set_username(username):
    client = gconf.client_get_default()
    client.set_string(GSM_USERNAME_PATH, username)


def set_password(password):
    client = gconf.client_get_default()
    client.set_string(GSM_PASSWORD_PATH, password)


def set_number(number):
    client = gconf.client_get_default()
    client.set_string(GSM_NUMBER_PATH, number)


def set_apn(apn):
    client = gconf.client_get_default()
    client.set_string(GSM_APN_PATH, apn)


def set_pin(pin):
    client = gconf.client_get_default()
    client.set_string(GSM_PIN_PATH, pin)


def set_puk(puk):
    client = gconf.client_get_default()
    client.set_string(GSM_PUK_PATH, puk)


def has_providers_db():
    if not os.path.isfile(COUNTRY_CODES_PATH):
        logging.debug("Mobile broadband provider database: Country " \
                          "codes path %s not found.", COUNTRY_CODES_PATH)
        return False
    try:
        tree = ElementTree(file=PROVIDERS_PATH)
    except (IOError, SyntaxError), e:
        logging.debug("Mobile broadband provider database: Could not read " \
                          "provider information %s error=%s", PROVIDERS_PATH)
        return False
    else:
        elem = tree.getroot()
        if elem is None or elem.get('format') != PROVIDERS_FORMAT_SUPPORTED:
            logging.debug("Mobile broadband provider database: Could not " \
                          "read provider information. %s is wrong format.",
                          elem.get('format'))
            return False
        return True


class CountryListStore(gtk.ListStore):
    COUNTRY_CODE = locale.getdefaultlocale()[0][3:5].lower()

    def __init__(self):
        gtk.ListStore.__init__(self, str, object)
        codes = {}
        with open(COUNTRY_CODES_PATH) as codes_file:
            for line in codes_file:
                if line.startswith('#'):
                    continue
                code, name = line.split('\t')[:2]
                codes[code.lower()] = name.strip()
        etree = ElementTree(file=PROVIDERS_PATH).getroot()
        self._country_idx = None
        i = 0
        for elem in etree.findall('.//country'):
            code = elem.attrib['code']
            if code == self.COUNTRY_CODE:
                self._country_idx = i
            else:
                i += 1
            if code in codes:
                self.append((codes[code], elem))
            else:
                self.append((code, elem))

    def get_row_providers(self, row):
        return self[row][1]

    def guess_country_row(self):
        if self._country_idx is not None:
            return self._country_idx
        else:
            return -1


class ProviderListStore(gtk.ListStore):
    def __init__(self, elem):
        gtk.ListStore.__init__(self, str, object)
        for provider_elem in elem.findall('.//provider'):
            apns = provider_elem.findall('.//apn')
            if not apns:
                # Skip carriers with CDMA entries only
                continue
            self.append((provider_elem.find('.//name').text, apns))

    def get_row_plans(self, row):
        return self[row][1]


class PlanListStore(gtk.ListStore):
    LANG_NS_ATTR = '{http://www.w3.org/XML/1998/namespace}lang'
    LANG = locale.getdefaultlocale()[0][:2]
    DEFAULT_NUMBER = '*99#'

    def __init__(self, elems):
        gtk.ListStore.__init__(self, str, object)
        for apn_elem in elems:
            plan = {}
            names = apn_elem.findall('.//name')
            if names:
                for name in names:
                    if name.get(self.LANG_NS_ATTR) is None:
                        # serviceproviders.xml default value
                        plan['name'] = name.text
                    elif name.get(self.LANG_NS_ATTR) == self.LANG:
                        # Great! We found a name value for our locale!
                        plan['name'] = name.text
                        break
            else:
                plan['name'] = _('Default')
            plan['apn'] = apn_elem.get('value')
            user = apn_elem.find('.//username')
            if user is not None:
                plan['username'] = user.text
            else:
                plan['username'] = ''
            passwd = apn_elem.find('.//password')
            if passwd is not None:
                plan['password'] = passwd.text
            else:
                plan['password'] = ''

            plan['number'] = self.DEFAULT_NUMBER

            self.append((plan['name'], plan))

    def get_row_plan(self, row):
        return self[row][1]
