#!/usr/bin/env python
# jhbuild - a build script for GNOME 1.x and 2.x
# Copyright (C) 2001-2006  James Henstridge
#
#   main.py: parses command line arguments and starts the build
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import errno
import logging
import optparse
import os
import sys

import gettext
import __builtin__
__builtin__.__dict__['N_'] = lambda x: x

import jhbuild.commands
from jhbuild.errors import UsageError, FatalError
from jhbuild.moduleset import warn_local_modulesets

import jhbuild.main

import bundlemodule
from config import Config
import depscheck


if sys.platform == 'darwin':
    # work around locale.getpreferredencoding() returning an empty string in
    # Mac OS X, see http://bugzilla.gnome.org/show_bug.cgi?id=534650 and
    # http://bazaar-vcs.org/DarwinCommandLineArgumentDecoding
    sys.platform = 'posix'
    try:
        import locale
    finally:
        sys.platform = 'darwin'
else:
    import locale

try:
    _encoding = locale.getpreferredencoding()
    assert _encoding
except (locale.Error, AssertionError):
    _encoding = 'ascii'


# Need a literal copy as there's no other way to use our own Config class :-/
def main(base_dir, args):
    gettext.install('jhbuild', unicode=True)

    logging.getLogger().setLevel(logging.INFO)
    logging_handler = logging.StreamHandler()
    logging_handler.setFormatter(jhbuild.main.LoggingFormatter())
    logging.getLogger().addHandler(logging_handler)
    parser = optparse.OptionParser(
        usage=_('%prog [ -f config ] command [ options ... ]'),
        add_help_option=False,
        description=_('Build a set of modules from diverse repositories in correct dependency order (such as GNOME).'))
    parser.disable_interspersed_args()
    parser.add_option('-h', '--help', action='callback',
                      callback=lambda *args: jhbuild.main.print_help(parser),
                      help=_("Display this help and exit"))
    parser.add_option('--help-commands', action='callback',
                      callback=lambda *args: jhbuild.main.print_help(parser),
                      help=optparse.SUPPRESS_HELP)
    parser.add_option('-f', '--file', action='store', metavar='CONFIG',
                      type='string', dest='configfile',
                      default=os.path.join(base_dir, 'sugar.jhbuildrc'),
                      help=_('use a non default configuration file'))
    parser.add_option('-m', '--moduleset', action='store', metavar='URI',
                      type='string', dest='moduleset', default=None,
                      help=_('use a non default module set'))
    parser.add_option('--no-interact', action='store_true',
                      dest='nointeract', default=False,
                      help=_('do not prompt for input'))

    options, args = parser.parse_args(args)

    try:
        config = Config(base_dir, options.configfile)
    except FatalError, exc:
        sys.stderr.write('sugar-jhbuild: %s\n' % exc.args[0].encode(_encoding, 'replace'))
        sys.exit(1)

    if options.moduleset: config.moduleset = options.moduleset
    if options.nointeract: config.interact = False

    if not args or args[0][0] == '-':
        command = 'build' # default to cvs update + compile
    else:
        command = args[0]
        args = args[1:]

    warn_local_modulesets(config)

    try:
        rc = jhbuild.commands.run(command, config, args, help=lambda: jhbuild.main.print_help(parser))
    except UsageError, exc:
        sys.stderr.write('sugar-jhbuild %s: %s\n' % (command, exc.args[0].encode(_encoding, 'replace')))
        parser.print_usage()
        sys.exit(1)
    except FatalError, exc:
        sys.stderr.write('sugar-jhbuild %s: %s\n' % (command, exc.args[0].encode(_encoding, 'replace')))
        sys.exit(1)
    except KeyboardInterrupt:
        uprint(_('Interrupted'))
        sys.exit(1)
    except EOFError:
        uprint(_('EOF'))
        sys.exit(1)
    except IOError, e:
        if e.errno != errno.EPIPE:
            raise
        sys.exit(0)
    if rc:
        sys.exit(rc)
