# Copyright (C) 2008, OLPC
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gtk
import gobject
from gettext import gettext as _

from sugar.graphics import style
from sugar.graphics.alert import Alert
from sugar.graphics.icon import Icon

from jarabe.controlpanel.sectionview import SectionView
from jarabe.controlpanel.inlinealert import InlineAlert


CLASS = 'Network'
ICON = 'module-network'
TITLE = _('Network')

_APPLY_TIMEOUT = 3000

class AddRemoveWidget(gtk.HBox):

    def __init__(self, label, add_button_clicked_cb,
                 remove_button_clicked_cb, index):
        gtk.Box.__init__(self)
        self.set_homogeneous(False)
        self.set_spacing(10)

        self._index = index
        self._add_button_added = False
        self._remove_button_added = False

        self._entry_box = gtk.Entry()
        self._entry_box.set_text(label)
        self.pack_start(self._entry_box, expand=False)
        self._entry_box.show()

        add_icon = Icon(icon_name='list-add')
        self._add_button = gtk.Button()
        self._add_button.set_image(add_icon)
        self._add_button.connect('clicked',
                                 add_button_clicked_cb,
                                 self)

        remove_icon = Icon(icon_name='list-remove')
        self._remove_button = gtk.Button()
        self._remove_button.set_image(remove_icon)
        self._remove_button.connect('clicked',
                                    remove_button_clicked_cb,
                                    self)

        self.__add_add_button()
        self.__add_remove_button()

    def _get_index(self):
        return self._index

    def _set_index(self, value):
        self._index = value

    def _get_entry(self):
        return self._entry_box.get_text()

    def __add_add_button(self):
        self.pack_start(self._add_button, expand=False)
        self._add_button.show()
        self._add_button_added = True

    def _remove_remove_button_if_not_already(self):
        if self._remove_button_added:
            self.__remove_remove_button()

    def __remove_remove_button(self):
        self.remove(self._remove_button)
        self._remove_button_added = False

    def _add_remove_button_if_not_already(self):
        if not self._remove_button_added:
            self.__add_remove_button()

    def __add_remove_button(self):
        self.pack_start(self._remove_button, expand=False)
        self._remove_button.show()
        self._remove_button_added = True


class MultiWidget(gtk.VBox):

    def __init__(self):
        gtk.VBox.__init__(self)

    def _add_widget(self, label):
        new_widget = AddRemoveWidget(label,
                                     self.__add_button_clicked_cb,
                                     self.__remove_button_clicked_cb,
                                     len(self.get_children()))
        self.add(new_widget)
        new_widget.show()
        self.show()
        self._update_remove_button_statuses()

    def __add_button_clicked_cb(self, add_button,
                                      add_button_container):
        self._add_widget('')
        self._update_remove_button_statuses()

    def __remove_button_clicked_cb(self, remove_button,
                                   remove_button_container):
        for child in self.get_children():
            if child._get_index() > remove_button_container._get_index():
                child._set_index(child._get_index() - 1)

        self.remove(remove_button_container)
        self._update_remove_button_statuses()

    def _update_remove_button_statuses(self):
        children = self.get_children()

        # Now, if there is only one entry, remove-button
        # should not be shown.
        if len(children) == 1:
            children[0]._remove_remove_button_if_not_already()

        # Alternatively, if there are more than 1 entries,
        # remove-button should be shown for all.
        if len(children) > 1:
            for child in children:
                child._add_remove_button_if_not_already()


    def _get_entries(self):
        entries = []
        for child in self.get_children():
            entries.append(child._get_entry())

        return entries


class Network(SectionView):
    def __init__(self, model, alerts):
        SectionView.__init__(self)

        self._model = model
        self.restart_alerts = alerts
        self._jabber_sid = 0
        self._jabber_valid = True
        self._radio_valid = True
        self._jabber_change_handler = None
        self._radio_change_handler = None
        self._network_configuration_reset_handler = None

        self.set_border_width(style.DEFAULT_SPACING * 2)
        self.set_spacing(style.DEFAULT_SPACING)
        group = gtk.SizeGroup(gtk.SIZE_GROUP_HORIZONTAL)

        self._radio_alert_box = gtk.HBox(spacing=style.DEFAULT_SPACING)
        self._jabber_alert_box = gtk.HBox(spacing=style.DEFAULT_SPACING)
        self._hidden_network_alert_box = gtk.HBox(spacing=style.DEFAULT_SPACING)

        workspace = gtk.VBox()
        workspace.show()

        separator_wireless = gtk.HSeparator()
        workspace.pack_start(separator_wireless, expand=False)
        separator_wireless.show()

        label_wireless = gtk.Label(_('Wireless'))
        label_wireless.set_alignment(0, 0)
        workspace.pack_start(label_wireless, expand=False)
        label_wireless.show()
        box_wireless = gtk.VBox()
        box_wireless.set_border_width(style.DEFAULT_SPACING * 2)
        box_wireless.set_spacing(style.DEFAULT_SPACING)

        radio_info = gtk.Label(_('Turn off the wireless radio to save battery'
                                 ' life'))
        radio_info.set_alignment(0, 0)
        radio_info.set_line_wrap(True)
        radio_info.show()
        box_wireless.pack_start(radio_info, expand=False)

        box_radio = gtk.HBox(spacing=style.DEFAULT_SPACING)
        self._button = gtk.CheckButton()
        self._button.set_alignment(0, 0)
        box_radio.pack_start(self._button, expand=False)
        self._button.show()

        label_radio = gtk.Label(_('Radio'))
        label_radio.set_alignment(0, 0.5)
        box_radio.pack_start(label_radio, expand=False)
        label_radio.show()

        box_wireless.pack_start(box_radio, expand=False)
        box_radio.show()

        self._radio_alert = InlineAlert()
        self._radio_alert_box.pack_start(self._radio_alert, expand=False)
        box_radio.pack_end(self._radio_alert_box, expand=False)
        self._radio_alert_box.show()
        if 'radio' in self.restart_alerts:
            self._radio_alert.props.msg = self.restart_msg
            self._radio_alert.show()

        history_info = gtk.Label(_('Discard network history if you have'
                                   ' trouble connecting to the network'))
        history_info.set_alignment(0, 0)
        history_info.set_line_wrap(True)
        history_info.show()
        box_wireless.pack_start(history_info, expand=False)

        box_clear_history = gtk.HBox(spacing=style.DEFAULT_SPACING)
        self._clear_history_button = gtk.Button()
        self._clear_history_button.set_label(_('Discard network history'))
        box_clear_history.pack_start(self._clear_history_button, expand=False)
        if not self._model.have_networks():
            self._clear_history_button.set_sensitive(False)
        self._clear_history_button.show()
        box_wireless.pack_start(box_clear_history, expand=False)
        box_clear_history.show()

        workspace.pack_start(box_wireless, expand=False)
        box_wireless.show()

        separator_mesh = gtk.HSeparator()
        workspace.pack_start(separator_mesh, False)
        separator_mesh.show()

        label_mesh = gtk.Label(_('Collaboration'))
        label_mesh.set_alignment(0, 0)
        workspace.pack_start(label_mesh, expand=False)
        label_mesh.show()
        box_mesh = gtk.VBox()
        box_mesh.set_border_width(style.DEFAULT_SPACING * 2)
        box_mesh.set_spacing(style.DEFAULT_SPACING)

        server_info = gtk.Label(_("The server is the equivalent of what"
                                  " room you are in; people on the same server"
                                  " will be able to see each other, even when"
                                  " they aren't on the same network."))
        server_info.set_alignment(0, 0)
        server_info.set_line_wrap(True)
        box_mesh.pack_start(server_info, expand=False)
        server_info.show()

        box_server = gtk.HBox(spacing=style.DEFAULT_SPACING)
        label_server = gtk.Label(_('Server:'))
        label_server.set_alignment(1, 0.5)
        label_server.modify_fg(gtk.STATE_NORMAL,
                               style.COLOR_SELECTION_GREY.get_gdk_color())
        box_server.pack_start(label_server, expand=False)
        group.add_widget(label_server)
        label_server.show()
        self._entry = gtk.Entry()
        self._entry.set_alignment(0)
        self._entry.modify_bg(gtk.STATE_INSENSITIVE,
                        style.COLOR_WHITE.get_gdk_color())
        self._entry.modify_base(gtk.STATE_INSENSITIVE,
                          style.COLOR_WHITE.get_gdk_color())
        self._entry.set_size_request(int(gtk.gdk.screen_width() / 3), -1)
        box_server.pack_start(self._entry, expand=False)
        self._entry.show()
        box_mesh.pack_start(box_server, expand=False)
        box_server.show()

        self._jabber_alert = InlineAlert()
        label_jabber_error = gtk.Label()
        group.add_widget(label_jabber_error)
        self._jabber_alert_box.pack_start(label_jabber_error, expand=False)
        label_jabber_error.show()
        self._jabber_alert_box.pack_start(self._jabber_alert, expand=False)
        box_mesh.pack_end(self._jabber_alert_box, expand=False)
        self._jabber_alert_box.show()
        if 'jabber' in self.restart_alerts:
            self._jabber_alert.props.msg = self.restart_msg
            self._jabber_alert.show()

        workspace.pack_start(box_mesh, expand=False)
        box_mesh.show()

        if self._model.is_hidden_network_connect_package_available():
            separator_hidden_network = gtk.HSeparator()
            workspace.pack_start(separator_hidden_network, False)
            separator_hidden_network.show()

            label_hidden_network = gtk.Label(_('Hidden Networks'))
            label_hidden_network.set_alignment(0, 0)
            workspace.pack_start(label_hidden_network, expand=False)
            label_hidden_network.show()
            box_hidden_network = gtk.VBox()
            box_hidden_network.set_border_width(style.DEFAULT_SPACING * 2)
            box_hidden_network.set_spacing(style.DEFAULT_SPACING)

            info = gtk.Label(_("Enter the SSIDs of the hidden networks."))
            info.set_alignment(0, 0)
            info.set_line_wrap(True)
            box_hidden_network.pack_start(info, expand=False)
            info.show()

            self._hidden_network_alert = InlineAlert()
            self.hidden_network_msg = 'XO-reboot is required for the' \
                                      ' changes to take effect.'
            self._hidden_network_alert.props.msg =_(self.hidden_network_msg)
            self._hidden_network_alert_box.pack_start(self._hidden_network_alert,
                                                        expand=False)
            box_hidden_network.pack_end(self._hidden_network_alert_box,
                                                expand=False)
            self._hidden_network_alert_box.show()
            self._hidden_network_alert.show()

            self._widget_table = MultiWidget()
            box_hidden_network.pack_start(self._widget_table, expand=False)
            self._widget_table.show()

            save_button = gtk.Button()
            save_button.set_alignment(0, 0)
            save_button.set_label('Save')
            save_button.connect('clicked', self.__save_button_clicked_cb)
            box_save_button = gtk.HBox()
            box_save_button.set_homogeneous(False)
            box_save_button.pack_start(save_button, expand=False)
            save_button.show()

            box_hidden_network.pack_start(box_save_button, expand=False)
            box_save_button.show()

            workspace.pack_start(box_hidden_network, expand=False)
            box_hidden_network.show()

        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        scrolled.add_with_viewport(workspace)
        scrolled.show()
        self.add(scrolled)

        self.setup()

    def setup(self):
        self._entry.set_text(self._model.get_jabber())
        try:
            radio_state = self._model.get_radio()
        except self._model.ReadError, detail:
            self._radio_alert.props.msg = detail
            self._radio_alert.show()
        else:
            self._button.set_active(radio_state)

        self._jabber_valid = True
        self._radio_valid = True
        self.needs_restart = False
        self._radio_change_handler = self._button.connect( \
                'toggled', self.__radio_toggled_cb)
        self._jabber_change_handler = self._entry.connect( \
                'changed', self.__jabber_changed_cb)
        self._network_configuration_reset_handler =  \
                self._clear_history_button.connect( \
                        'clicked', self.__network_configuration_reset_cb)

        if self._model.is_hidden_network_connect_package_available():

            ssids = self._model.get_ssids()
            if len(ssids) == 0:
                self._widget_table._add_widget('')
            else:
                for ssid in ssids:
                    self._widget_table._add_widget(ssid)

    def undo(self):
        self._button.disconnect(self._radio_change_handler)
        self._entry.disconnect(self._jabber_change_handler)
        self._model.undo()
        self._jabber_alert.hide()
        self._radio_alert.hide()

    def _validate(self):
        if self._jabber_valid and self._radio_valid:
            self.props.is_valid = True
        else:
            self.props.is_valid = False

    def __radio_toggled_cb(self, widget, data=None):
        radio_state = widget.get_active()
        try:
            self._model.set_radio(radio_state)
        except self._model.ReadError, detail:
            self._radio_alert.props.msg = detail
            self._radio_valid = False
        else:
            self._radio_valid = True
            if self._model.have_networks():
                self._clear_history_button.set_sensitive(True)

        self._validate()
        return False

    def __jabber_changed_cb(self, widget, data=None):
        if self._jabber_sid:
            gobject.source_remove(self._jabber_sid)
        self._jabber_sid = gobject.timeout_add(_APPLY_TIMEOUT,
                                               self.__jabber_timeout_cb,
                                               widget)

    def __jabber_timeout_cb(self, widget):
        self._jabber_sid = 0
        if widget.get_text() == self._model.get_jabber:
            return
        try:
            self._model.set_jabber(widget.get_text())
        except self._model.ReadError, detail:
            self._jabber_alert.props.msg = detail
            self._jabber_valid = False
            self._jabber_alert.show()
            self.restart_alerts.append('jabber')
        else:
            self._jabber_valid = True
            self._jabber_alert.hide()

        self._validate()
        return False

    def __network_configuration_reset_cb(self, widget):
        # FIXME: takes effect immediately, not after CP is closed with
        # confirmation button
        self._model.clear_networks()
        if not self._model.have_networks():
            self._clear_history_button.set_sensitive(False)

    def __save_button_clicked_cb(self, save_button):
        self._model.set_ssids(self._widget_table._get_entries())
        save_button.set_sensitive(False)
