# Copyright (C) 2010, Walter Bender, Sugar Labs
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


from gettext import gettext as _
import os

import gtk
import gconf

from sugar.graphics.tray import TrayIcon
from sugar.graphics.xocolor import XoColor
from sugar.graphics.palette import Palette
from sugar.graphics import style

from jarabe.frame.frameinvoker import FrameWidgetInvoker

TOUCHPAD_MODES = ['capacitive', 'resistive']
STATUS_TEXT = {TOUCHPAD_MODES[0]: _('finger'), TOUCHPAD_MODES[1]: _('stylus')}
STATUS_ICON = {TOUCHPAD_MODES[0]: 'touchpad-' + TOUCHPAD_MODES[0],
               TOUCHPAD_MODES[1]: 'touchpad-' + TOUCHPAD_MODES[1]}
# FLAG_PATH is used to preserve status between boots.
FLAG_PATH = '/home/olpc/.olpc-pentablet-mode'
# NODE_PATH is used to communicate with the touchpad device.
NODE_PATH = '/sys/devices/platform/i8042/serio1/ptmode'

_view = None

class DeviceView(TrayIcon):
    """ Manage the touchpad mode from the device palette on the Frame. """

    FRAME_POSITION_RELATIVE = 500

    def __init__(self):
        """ Create the touchpad palette and display it on Frame. """
        icon_name = STATUS_ICON[read_touchpad_mode()]

        client = gconf.client_get_default()
        color = XoColor(client.get_string('/desktop/sugar/user/color'))
        TrayIcon.__init__(self, icon_name=icon_name, xo_color=color)

        self.set_palette_invoker(FrameWidgetInvoker(self))
        self.connect('button-release-event', self.__button_release_event_cb)

    def create_palette(self):
        """ On create, set the current mode. """
        self._palette = ResourcePalette(_('My touchpad'), self.icon)
        self._palette.set_group_id('frame')
        return self._palette

    def __button_release_event_cb(self, widget, event):
        """ On button release, switch modes. """
        self._palette.toggle_mode()
        return True


class ResourcePalette(Palette):
    """ Query the current state of the touchpad and update the display. """

    def __init__(self, primary_text, icon):
        """ Create the palette and initilize with current touchpad status. """
        Palette.__init__(self, label=primary_text)

        self._icon = icon

        vbox = gtk.VBox()
        self.set_content(vbox)

        self._status_text = gtk.Label()
        vbox.pack_start(self._status_text, padding=style.DEFAULT_PADDING)
        self._status_text.show()

        vbox.show()

        self._mode = read_touchpad_mode()
        self._update()

    def _update(self):
        """ Update the label and icon based on the current mode. """
        self._status_text.set_label(STATUS_TEXT[self._mode])
        self._icon.props.icon_name = STATUS_ICON[self._mode]

    def toggle_mode(self):
        """ On mouse click, toggle the mode. """
        self._mode = TOUCHPAD_MODES[1 - TOUCHPAD_MODES.index(self._mode)]
        write_touchpad_mode(self._mode)
        self._update()


def setup(tray):
    """ Touchpad palette only appears when the device exisits. """
    global _view
    if os.path.exists(NODE_PATH):
        _view = DeviceView()
        tray.add_device(_view)


def read_touchpad_mode():
    """ Read the touchpad mode from the node path. """
    node_file_handle = open(NODE_PATH, 'r')
    text = node_file_handle.read()
    node_file_handle.close()

    return TOUCHPAD_MODES[int(text[0])]


def write_touchpad_mode(touchpad):
    """ Write the touchpad mode to the node path and set/unset the flag. """
    touchpad_mode_index = TOUCHPAD_MODES.index(touchpad)

    node_file_handle = open(NODE_PATH, 'w')
    node_file_handle.write(str(touchpad_mode_index))
    node_file_handle.close()

    if touchpad_mode_index == 0:
        if os.path.exists(FLAG_PATH):
            os.remove(FLAG_PATH)
    else:
        flag_file_handle = open(FLAG_PATH, 'w')
        flag_file_handle.close()
