# Copyright (C) Mukesh Gupta <mukeshgupta.2006@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import logging
from gettext import gettext as _

import gconf
import gtk

from sugar import profile
from sugar.graphics import style
from sugar.graphics.icon import Icon
from sugar.graphics.tray import TrayIcon
from sugar.graphics.palette import Palette
from sugar.graphics.menuitem import MenuItem
from sugar.graphics.toolbutton import ToolButton

from jarabe.model import feedback_collector


_ICON_NAME = 'feedback-icon'


class DeviceView(TrayIcon):

    FRAME_POSITION_RELATIVE = 100

    def __init__(self):
        TrayIcon.__init__(self, icon_name=_ICON_NAME,
                xo_color=profile.get_color())
        self.create_palette()

    def create_palette(self):
        logging.debug('palette created')
        self.palette = _Palette(_('Feedback'))
        self.palette.set_group_id('frame')
        return self.palette


class _Palette(Palette):

    def __init__(self, primary_text):
        Palette.__init__(self, primary_text)

        icon = Icon()
        icon.set_from_icon_name('emblem-favorite', gtk.ICON_SIZE_MENU)
        icon.props.xo_color = profile.get_color()

        personalized = MenuItem(_('Personalized submit...'))
        personalized.set_image(icon)
        personalized.connect('activate', self.__personalized_activate_cb)
        personalized.show()
        self.menu.append(personalized)

        self._anonymous = MenuItem(_('Anonymous submit'), 'emblem-favorite')
        self._anonymous.connect('activate', self.__anonymous_activate_cb)
        self._anonymous.show()
        self.menu.append(self._anonymous)

    def popup(self, immediate=False, state=None):
        self._anonymous.set_sensitive(not feedback_collector.is_empty())
        Palette.popup(self, immediate=immediate, state=state)

    def __anonymous_activate_cb(self, button):
        feedback_collector.anonymous_submit()

    def __personalized_activate_cb(self, button):
        window = _Window()
        window.show()


class _Window(gtk.Window):

    __gtype_name__ = 'FeedbackWindow'

    def __init__(self):
        gtk.Window.__init__(self)

        self.set_border_width(style.LINE_WIDTH)
        offset = style.GRID_CELL_SIZE
        width = gtk.gdk.screen_width() - offset * 2
        height = gtk.gdk.screen_height() - offset * 2
        self.set_size_request(width, height)
        self.set_position(gtk.WIN_POS_CENTER_ALWAYS)
        self.set_decorated(False)
        self.set_resizable(False)
        self.set_modal(True)

        canvas = gtk.VBox()
        self.add(canvas)

        self._toolbar = gtk.Toolbar()
        canvas.pack_start(self._toolbar, False)

        icon = Icon()
        icon.set_from_icon_name('emblem-favorite', gtk.ICON_SIZE_LARGE_TOOLBAR)
        icon.props.xo_color = profile.get_color()
        self._add_widget(icon)

        self._add_separator(False)

        title = gtk.Label(_('Submit feedback with contact infromation'))
        self._add_widget(title)

        self._add_separator(True)

        submit = ToolButton('dialog-ok', tooltip=_('Submit'))
        submit.connect('clicked', lambda button: self._submit())
        self._toolbar.insert(submit, -1)

        cancel = ToolButton('dialog-cancel', tooltip=_('Cancel'))
        cancel.connect('clicked', lambda button: self.destroy())
        self._toolbar.insert(cancel, -1)

        bg = gtk.EventBox()
        bg.modify_bg(gtk.STATE_NORMAL, style.COLOR_WHITE.get_gdk_color())
        canvas.pack_start(bg)

        self._message = gtk.TextView()
        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        scrolled.set_border_width(style.DEFAULT_PADDING)
        scrolled.add(self._message)
        bg.add(scrolled)

        self.show_all()
        self.set_focus(self._message)

        self.connect("realize", self.__realize_cb)

    def do_key_press_event(self, event):
        if event.keyval == gtk.keysyms.Escape:
            self.destroy()
        elif event.keyval == gtk.keysyms.Return and \
                event.state & gtk.gdk.CONTROL_MASK:
            self._submit()
        else:
            gtk.Window.do_key_press_event(self, event)

    def _add_widget(self, widget):
        tool_item = gtk.ToolItem()
        tool_item.add(widget)
        self._toolbar.insert(tool_item, -1)

    def _add_separator(self, expand):
        separator = gtk.SeparatorToolItem()
        separator.props.draw = False
        if expand:
            separator.set_expand(True)
        else:
            separator.set_size_request(style.DEFAULT_SPACING, -1)
        self._toolbar.insert(separator, -1)

    def _submit(self):
        feedback_collector.submit(self._message.props.buffer.props.text)
        self.destroy()

    def __realize_cb(self, widget):
        self.window.set_type_hint(gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
        self.window.set_accept_focus(True)


def setup(tray):
    client = gconf.client_get_default()
    if client.get_bool('/desktop/sugar/feedback/enabled'):
        tray.add_device(DeviceView())
