/*! This file has been prepared for Doxygen automatic documentation generation.
 *	\file motor.h
 *	\brief Utilities and functions for controlling the stepper motor
 *	
 *	This file contains statements and functions to control the behaviour
 *	of the stepper motor.
 */
 
 #ifndef __MOTOR_C
	#define __MOTOR_C
	
	/*!	\var unsigned char speed
	 *	\brief Prescaling of PLL clock
	 */
	unsigned char speed = '1';
	
	/*!	\var unsigned char pwmstate
	 *	\brief current value of PWM duty cycle reffered from the sine table
	 */
	unsigned char pwmstate = 0;
	

	int i=0;
	
	/*!	\var unsigned char home
	 *	\brief Status of home position
	 */	
	unsigned char home='0';
	
	unsigned char in=0;
	
	/*!	\var unsigned char curr_command
	 *	\brief Current command being executed.
	 */	
	unsigned char curr_command = 'N';	//N=None
	
	/*!	\var unsigned char major_step
	 *	\brief major steps left to move
	 */	
	unsigned char major_step=0;
	
	/*!	\var unsigned char revolutions
	 *	\brief revolutions left to move
	 */	
	unsigned char revolutions =0;
	
	/*!	\var unsigned char minor_step
	 *	\brief minor steps left to move
	 */	
	unsigned char minor_step=0;
	
	unsigned char step_started=0;
	
	unsigned char major_recv=0;
	
	/*!	\var char step
	 *	\brief 	traverse by 'step' in sin table at next 
	 *			timer1 overflow interrupt.
	 */	
	char step=1;
	
	unsigned char temp=0;

	unsigned char diff=0;
	
	/*!	\fn static void velocity(void)
	 *	\brief velocity/speed/PWM_frequency control
	 *	
	 *	Accepts a number(from UART) between '1' to '9' 
	 *	'1' being the highest and '9' the slowest speed 
	 *	Each speed is half its immediately lesser value 
	 *	so '9' is 1/256th of the fastest available(which is'1')
	 *
	 *	\return void
	 */
	void velocity()
	{
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		speed=recv_char();
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		if(speed<'1')
			speed='1';
		if(speed>'9')
			speed='9';		
		_ENABLE_TIMER1_OVERFLOW_INTERRUPT();
	}
	
	/*!	\fn static void forward(void)
	 *	\brief Move the motor in forward direction
	 *	
	 *	Accepts 3 values from UART: \n
	 *	a) No. of complete revolutions to move forward. \n
	 *	b) No. of major steps to move forward.          \n
	 *	c) No. of minor steps to move forward.          \n\n
	 *	Note: 1 complete revolution	=  50  major steps  \n
	 *	      1 major step          =  256 minor steps  \n
	 *        1 step of stepper     =  64  minor steps  =  1/4 major step \n
	 *	Each of these 3 values is accepted as an 8 bit unsigned value \n
	 *
	 *	\return void
	 */
	void forward()
	{
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		revolutions = recv_char();
		major_step=recv_char();
		major_recv=major_step;
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		temp=recv_char();
		if((i+temp)>255)
		{
			i+=temp;
			i-=256;
			temp=minor_step;
			minor_step=i;
			i=temp;
		}
		else
			minor_step+=temp;
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		step=1;
		step_started=0;
		curr_command='F';
		_ENABLE_TIMER1_OVERFLOW_INTERRUPT();
	}
	 
	/*!	\fn static void reverse_effect(void)
	 *	\brief Make index conversions to map 'sin to cos' for same value.
	 *	
	 *	Adjusts the current position of the index into the sine table
	 *	so that if it is currently in the increasing part of the curve,
	 *	 the new value is in the decreasing part , and vice versa.
	 *	This is required to be done whenever the motor has to reverse its
	 *	direction of motion.
	 *
	 *	\return void
	 */
	void reverse_effect()
	{
		if(i<=64)
		{
			if(i<=32)
			{
				diff=32-i;
				i+=2*diff;
			}
			else
			{
				diff=i-32;
				i-=2*diff;
			}
		}
		else
		{
			if(i<160)
			{
				diff=160-i;
				i+=2*diff;
			}
			else
			{
				diff=i-160;
				i-=2*diff;
			}
		}
	}
 
	/*!	\fn static void reverse(void)
	 *	\brief Move the motor in reverse direction
	 *	
	 *	Accepts 3 values from UART: \n
	 *	a) No. of complete revolutions to move reverse. \n
	 *	b) No. of major steps to move reverse.          \n
	 *	c) No. of minor steps to move reverse.          \n\n
	 *	Note: 1 complete revolution	=  50  major steps  \n
	 *	      1 major step          =  256 minor steps  \n
	 *        1 step of stepper     =  64  minor steps  =  1/4 major step \n
	 *	Each of these 3 values is accepted as an 8 bit unsigned value \n
	 *
	 *	\return void
	 */
	void reverse()
	{
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		revolutions=recv_char();
		major_step=recv_char();
		major_recv=major_step;
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		temp=recv_char();
		reverse_effect();
		minor_step=i;
		if((i+temp)>255)
		{
			i+=temp;
			i-=256;
			temp=minor_step;
			minor_step=i;
			i=temp;
		}
		else
			minor_step+=temp;		
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		step=1;
		step_started=0;
		curr_command='R';
		_ENABLE_TIMER1_OVERFLOW_INTERRUPT();
	}

	/*!	\fn static void to_home(void)
	 *	\brief Move the motor to home position #1 or #2
	 *	
	 *	Accepts a value from USART, which can be either '1' or '2': \n
	 *	1. '1' represents home position 1 : Stops moving when a low voltage
	 *	 is received on PB0. \n
	 *	2. '2' represents home position 2 : Stops moving when a low voltage
	 *	 is received on PB3. \n\n 
	 *	NOTE: In both home 1 and home 2, the stepper moves in forward
	 *	direction only.
	 *
	 *	\return void
	 */
	void to_home()
	{
		_DISABLE_TIMER1_OVERFLOW_INTERRUPT();
		home=recv_char()-'0';
		step=1;
		curr_command='H';
		_ENABLE_TIMER1_OVERFLOW_INTERRUPT();
	}

	/*!	\fn static void abort(void)
	 *	\brief Immediately abort the current operation.
	 *	
	 *	Aborts whichever command was last given. \n\n
	 *	Returns a D followed by an alphabet representing the command that
	 *	 was aborted. \n
	 *	e.g. If the last command was to move forward(F) then on executing 
	 *	abort DF would be sent.
	 *
	 *	\return void
	 */
	void abort()
	{
		send_char('D');
		send_char(curr_command);
		major_step=0;		//added on dec 25
		curr_command='N';	
	}

	/*!	\fn static void status(void)
	 *	\brief Previous command executed
	 *	
	 *	Returns the command which is currently being processed. 
	 *	If no command is being executed, a N is sent. \n
	 *	e.g. If the motor is currently moving in the reverse direction, 
	 *	an R is sent
	 *
	 *	\return void
	 */
	void status()
	{
		send_char(curr_command);	
	}
	
 #endif
