/*! This file has been prepared for Doxygen automatic documentation generation.
 *	\file config.h
 *	\brief This file contains many important system parameters and statements.
 *	
 *	This file implements a set of parameters that allow the system to be cofigured.
 * 	It contains a set of mnemonic parameters that make the system easy to understand.
 */

#ifndef __CONFIG_H

	#define __CONFIG_H
	
	/*!	\def _RESET_PORT
	 *	\brief Resets all pins in PORTB configuring each pin as an output pin. 
	 */
	#define _RESET_PORT()	DDRB=(1<<DDB5)|(1<<DDB4)|(1<<DDB3)|(1<<DDB2)|(1<<DDB1)|(1<<DDB0)
	
	/*!	\def _CONFIGURE_PWM1_CHANNEL()
	 *	\brief Initialize PWM1 channel.
	 *
	 *	Initialize PWM1 channel.(OC1A pin on the microcontroller) channel
	 *	setting the output compare register OCR1A to 'some' value.
	 */
	#define _CONFIGURE_PWM1_CHANNEL()	OCR1A	=	0x80
	
	
	/*!	\def _CONFIGURE_PWM2_CHANNEL()
	 *	\brief Initialize PWM2 channel.
	 *
	 *	Initialize PWM2 channel.(OC1B pin on the microcontroller) channel
	 *	setting the output compare register OCR1B to 'some' value.
	 */
	#define _CONFIGURE_PWM2_CHANNEL()	OCR1B	=	0x80
	
	/*!	\def _CONFIGURE_PWM_BASE_FREQ()
	 *	\brief Set OCR1C to TOP
	 *
	 *	Set the the output compare register OCR1Cto 255 (TOP).
	 *	This determines the PWM base frequency \n\n
	 *	The PWM base frequency can be calculated as: \n\n
	 *  \f[ f_w=\frac{(f_p)}{TOP+1} \f]
	 *	\n where, \n
	 *	\f$ f_w \f$ = The PWM base frequency, \n
	 *	\f$ f_p \f$ = Is the PLL clock frequency set by CS13/12/11/10 bits in TCCR1, \n
	 *	and, \f$ TOP = 255 \f$ \n\n
	 */
	#define _CONFIGURE_PWM_BASE_FREQ()	OCR1C=0xFF
	
	/*!	\def _CONFIGURE_I_O_PINS()
	 *	\brief Configure the GPIO pins on AtTiny45 according to the given system. 
	 *  
	 *
	 *  <C>
	 *	|Pin No.|___Description_________|___Config._| \n
	 *  _____________________________________________ \n	
	 *	|__PB0__|___home position #1____|___Input___| \n
	 *	|__PB1__|___PWM output #1_______|___Output__| \n
	 *	|__PB2__|___RXD of soft UART____|___Input___| \n
	 *	|__PB3__|___home position #2____|___Input___| \n
	 *	|__PB4__|___PWM output #2_______|___Output__| \n
	 *	|__PB5__|___TXD of soft UART____|___Output__| \n
	 *  _____________________________________________ \n
	 *  </C>
	 */
	#define _CONFIGURE_I_O_PINS()	DDRB=(1<<DDB5)|(1<<DDB4)|(0<<DDB3)|(0<<DDB2)|(1<<DDB1)|(0<<DDB0)
	
	/*!	\def _SET_INTERNAL_PULLUPS()
	 *	\brief Set the internal pullups of pins configured as 'Input'
	 *
	 *	This statement configures the internal pullup resistors of the GPIO pins 
	 *	configured as 'Input' by <I><B>setting</B></I> them. \n\n
	 *	This statement should be preceded by the _CONFIGURE_I_O_PINS() statement
	 */	
	#define _SET_INTERNAL_PULLUPS()	PORTB|=(0<<PB5)|(0<<PB4)|(1<<PB3)|(1<<PB2)|(0<<PB1)|(1<<PB0)
	
	/*!	\def _ENABLE_PWM1_CHANNEL()
	 *	\brief Selects and Enables PWM output on <B>OC1A</B> pin.
	 *
	 *	This statement selects and enables PWM output on the OC1A pin on Attiny45.
	 *	Additionally it sets the default frequency of the TCNT1 timer counter register.
	 */
	#define _ENABLE_PWM1_CHANNEL()	TCCR1=((1<<PWM1A)|(1<<COM1A1))
	
	/*!	\def _ENABLE_PWM2_CHANNEL()
	 *	\brief Selects and Enables PWM output on <B>OC1B</B> pin.
	 *
	 *	This statement selects and enables PWM output on the OC1B pin on Attiny45.
	 */
	#define _ENABLE_PWM2_CHANNEL()	GTCCR=(1<<PWM1B)|(1<<COM1B1)
	
	/*!	\def _ENABLE_PLL()
	 *	\brief Enable the PLL by setting PLLE bit in PLLCSR
	 *
	 *	The PLL is started and if needed internal RC-oscillator is started as a PLL reference clock.
	 */
	#define _ENABLE_PLL()	PLLCSR=(1<<PLLE)
	
	/*!	\def _WAIT_TILL_PLL_LOCKED()
	 *	\brief Wait till the PLL is locked
	 *
	 *	This statement configures the internal pullup resistors of the GPIO pins 
	 *	configured as 'Input' by <I><B>setting</B></I> them. \n\n
	 *	This statement should be preceded by the _CONFIGURE_I_O_PINS() statement
	 */
	#define _WAIT_TILL_PLL_LOCKED()	while(!(PLLCSR&(1<<PLOCK)));
	
	/*!	\def _DELAY_1000_CYCLES()
	 *	\brief A delay of 1000 clock cycles=125 microseconds at 8MHz
	 */
	#define _DELAY_1000_CYCLES()	for(unsigned int j=0;j<1000;j++)asm("NOP");
		
	/*!	\def _SET_INTERNAL_PULLUPS()
	 *	\brief Set the internal pullups of pins configured as 'Input'
	 *
	 *	When the PLOCK bit is set, the PLL is locked to the reference clock. The PLOCK bit should be
	 *	ignored during initial PLL lock-in sequence when PLL frequency overshoots and undershoots,
	 *	before reaching steady state. The steady state is obtained within 100 s. After PLL lock-in it is
	 *	recommended to check the PLOCK bit before enabling PCK for Timer/Counter1.
	 */
	#define _ENABLE_PLL_CLOCK_SOURCE()	PLLCSR|=(1<<PCKE)
	
	/*!	\def _ENABLE_GLOBAL_INTERRUPTS()
	 *	\brief Enable global interrupts.
	 *
	 *	Enables global interrupts by setting the I bit in SREG register
	 */
	#define _ENABLE_GLOBAL_INTERRUPTS()	asm("sei")
	
	/*!	\def _DISABLE_GLOBAL_INTERRUPTS()
	 *	\brief Disable global interrupts.
	 *
	 *	Disables global interrupts by clearing the I bit in SREG register
	 */
	#define _DISABLE_GLOBAL_INTERRUPTS()	asm("cli")
	
#endif
