/*! This file has been prepared for Doxygen automatic documentation generation.
 *	\file motor.h
 *	\brief Utilities and functions for controlling the stepper motor
 *	
 *	This file contains statements and functions to control the behaviour
 *	of the stepper motor.
 */
 
#ifndef __MOTOR_H
	
	#define __MOTOR_H
	
	/*!	\def _ENABLE_TIMER1_OVERFLOW_INTERRUPT()
	 *
	 *	\brief Enable TCCR1.
	 *
	 *  Enable the Timer/Counter1 Overflow Interrupt.
	 */
	#define _ENABLE_TIMER1_OVERFLOW_INTERRUPT() 	TIMSK|=(1<<TOIE1)
	
	/*! \def _DISABLE_TIMER1_OVERFLOW_INTERRUPT()
	 *	
	 *	\brief Disable TCCR1.
	 *
	 *  Disable the Timer/Counter1 Overflow Interrupt.
	 */
	#define _DISABLE_TIMER1_OVERFLOW_INTERRUPT() 	TIMSK&=(0<<TOIE1)
	
	/*!	_CONFIGURE_TIMER1_SET_SPEED_MODE()
	 *
	 *	\brief Configure the TCCR1 control register for PWM output.
	 *
	 *  Configures the TCCR1 control register to enable the PWM mode based 
	 *	on comparator OCR1A in Timer/Counter1. \n
	 *	Clears the OC1A output line after a compare match with OCR1C
	 *	register value.
	 */
	#define _CONFIGURE_TIMER1_SET_SPEED_MODE()	TCCR1=(1<<PWM1A)|(1<<COM1A0)
	
	/*!	\def _SET_TIMER1_FREQUENCY(n) 
	 *
	 *	\brief Configure the TCCR1 control register to set the timer 
	 *	frequency(n=1 to 7).
	 *
	 *  Configures the TCCR1 control register to define the prescaling
	 *	source of Timer/Counter1. 
	 *	It determines the speed at which the steeper motor will operate.
	 *	A lower value of n indicates a higher speed of operation
	 *	and vice versa. \n\n
	 *
	 *		<C>
	 *		|__n___|_CS13__|_CS12__|_CS11__|_CS10__|____Prescaling_|  \n
	 *		-------------------------------------------------------
	 *		|__1___|___0___|___1___|___0___|___0___|_____CK/8______| \n
	 *		|__2___|___0___|___1___|___0___|___1___|_____CK/16_____| \n
	 *		|__3___|___0___|___1___|___1___|___0___|_____CK/32_____| \n
	 *		|__4___|___0___|___1___|___1___|___1___|_____CK/64_____| \n
	 *		|__5___|___1___|___0___|___0___|___0___|_____CK/128____| \n
	 *		|__6___|___1___|___0___|___0___|___1___|_____CK/256____| \n
	 *		|__7___|___1___|___0___|___1___|___0___|_____CK/512____| \n
	 *		</C>
	 *	IMPORTANT: For smooth operation of the stepper motor, n must lie within this range (1-7).
	 *
	 *	 \param n TCCR1 timer frequency.
	 */
	#define _SET_TIMER1_FREQUENCY(n) 				TCCR1  |= n+3
	
	//Prototypes
	
	/*!	\fn static void velocity(void)
	 *	\brief velocity/speed/PWM_frequency control
	 *	
	 *	Accepts a number(from UART) between '1' to '7' 
	 *	'1' being the highest and '7' the slowest speed 
	 *	Each speed is half its immediately lesser value 
	 *	so '7' is 1/64th of the fastest available(which is'1')
	 *
	 *	\return void
	 */
	static void velocity(void);
	
	/*!	\fn static void forward(void)
	 *	\brief Move the motor in forward direction
	 *	
	 *	Accepts 3 values from UART: \n
	 *	a) No. of complete revolutions to move forward. \n
	 *	b) No. of major steps to move forward.          \n
	 *	c) No. of minor steps to move forward.          \n\n
	 *	Note: 1 complete revolution	=  50  major steps  \n
	 *	      1 major step          =  256 minor steps  \n
	 *        1 step of stepper     =  64  minor steps  =  1/4 major step \n
	 *	Each of these 3 values is accepted as an 8 bit unsigned value \n
	 *
	 *	\return void
	 */
	static void forward(void);
	
	/*!	\fn static void reverse_effect(void)
	 *	\brief Make index conversions to map 'sin to cos' for same value.
	 *	
	 *	Adjusts the current position of the index into the sine table
	 *	so that if it is currently in the increasing part of the curve,
	 *	 the new value is in the decreasing part , and vice versa.
	 *	This is required to be done whenever the motor has to reverse its
	 *	direction of motion.
	 *
	 *	\return void
	 */
	static void reverse_effect(void);
	
	/*!	\fn static void reverse(void)
	 *	\brief Move the motor in reverse direction
	 *	
	 *	Accepts 3 values from UART: \n
	 *	a) No. of complete revolutions to move reverse. \n
	 *	b) No. of major steps to move reverse.          \n
	 *	c) No. of minor steps to move reverse.          \n\n
	 *	Note: 1 complete revolution	=  50  major steps  \n
	 *	      1 major step          =  256 minor steps  \n
	 *        1 step of stepper     =  64  minor steps  =  1/4 major step \n
	 *	Each of these 3 values is accepted as an 8 bit unsigned value \n
	 *
	 *	\return void
	 */
	static void reverse(void);
	
	/*!	\fn static void to_home(void)
	 *	\brief Move the motor to home position #1 or #2
	 *	
	 *	Accepts a value from USART, which can be either '1' or '2': \n
	 *	1. '1' represents home position 1 : Stops moving when a low voltage
	 *	 is received on PB0. \n
	 *	2. '2' represents home position 2 : Stops moving when a low voltage
	 *	 is received on PB3. \n\n 
	 *	NOTE: In both home 1 and home 2, the stepper moves in forward
	 *	direction only.
	 *
	 *	\return void
	 */
	static void to_home(void);
	
	/*!	\fn static void abort(void)
	 *	\brief Immediately abort the current operation.
	 *	
	 *	Aborts whichever command was last given. \n\n
	 *	Returns a D followed by an alphabet representing the command that
	 *	 was aborted. \n
	 *	e.g. If the last command was to move forward(F) then on executing 
	 *	abort DF would be sent.
	 *
	 *	\return void
	 */
	static void abort(void);
	
	/*!	\fn static void status(void)
	 *	\brief Previous command executed
	 *	
	 *	Returns the command which is currently being processed. 
	 *	If no command is being executed, a N is sent. \n
	 *	e.g. If the motor is currently moving in the reverse direction, 
	 *	an R is sent
	 *
	 *	\return void
	 */
	static void status(void);
		
#endif
