/*! This file has been prepared for Doxygen automatic documentation generation.
 *	\file compiler.h
 *	\brief Common and compiler specific files.
 *	
 *	This files allows to incorporate both, AVR-GCC and IAR-C compilers to compile 
 *	the same piece of code. \n\n
 *	1.   Includes common and compiler specific header files. \n
 *	2.   Makes standard headers for some compiler specific ones.\n	
 *	 
 *	Compiler specific include files and #defines inclusion...
 *	\code 
 *	#ifdef _AVR_GCC
 *		#include <avr\io.h>
 *		#include <avr\interrupt.h>
 *		#include <avr\pgmspace.h>
 *		#define _SIN_TABLE(i)		pgm_read_byte(&sinewave[i])
 *		#define _COS_TABLE(i)		pgm_read_byte(&sinewave[(i+64)%256])
 *		#define _TIMER1_OVERFLOW_INTERRUPT() ISR(TIM1_OVF_vect)
 *	#endif
 *
 *	#ifdef _IAR_C
 *		#include <ioavr.h>
 *		#include <inavr.h>
 *		#include <pgmspace.h>
 *		#define _SIN_TABLE(i)		sinewave[i]
 *		#define _COS_TABLE(i)		sinewave[(i+64)%256]
 *		#define _TIMER1_OVERFLOW_INTERRUPT() (__interrupt void Timer1_Overflow_ISR(void))
 *	#endif
 *	\endcode
 *
 *	Including files commpon to both compilers.	
 *	\code 
 *	#include "motor.h"
 *	#include "config.h"
 *	#include "tables.h"
 *	#include "uart.h"
 *	#include "motor.c"
 *	#include "uart.c"
 *	\endcode  
 */

#ifndef __COMPILER_H

	#define __COMPILER_H
	
	/*!	\def _IAR_C
	 *	\brief Include Compiler specific code for <I>IAR C</I> compiler
	 *
	 *	If the code is to be compiled woth <I>IAR C</I> compiler the this 
	 *	statement should be left uncommented, and <C>#define _AVR_GCC</C>
	 *	should be commented.
	 */
	//#define _IAR_C
	
	/*!	\def _AVR_GCC
	 *	\brief Include Compiler specific code for <I>ACR-GCC</I> compiler
	 *
	 *	If the code is to be compiled woth <I>AVR-GCC</I> compiler the this 
	 *	statement should be left uncommented, and <C>#define _IAR_C</C>
	 *	should be commented.
	 */
	#define _AVR_GCC
	
	#ifdef _AVR_GCC
		#include <avr\io.h>
		#include <avr\interrupt.h>
		#include <avr\pgmspace.h>
		
		/*!	\def _SIN_TABLE(i)
		 *	\brief Lookup sine table.
		 *
		 *	Lookup the sine table stored in flash memory and access the ith element
		 *	
		 */
		#define _SIN_TABLE(i)	pgm_read_byte(&sinewave[i])
		
		/*!	\def _COS_TABLE(i)
		 *	\brief Lookup cos table.
		 *
		 *	Lookup the same sine table as in <I>#define _SIN_TABLE(i)</I> stored 
		 *	in flash memory,convert it into an effective cos table and retrieve
		 *	the value of the ith element in the effective cos table.	
		 */
		#define _COS_TABLE(i)	pgm_read_byte(&sinewave[(i+64)%256])
		
		/*!	\def _TIMER1_OVERFLOW_INTERRUPT()
		 *	\brief Compiler specific header for timer1 overflow ISR
		 *
		 *	This statement is neccecary for incorporating both AVR_GCC, and IAR_C
		 *	compilers. Defines a common 'TIMER1 overflow interrupt' header.
		 */
		#define _TIMER1_OVERFLOW_INTERRUPT()	ISR(TIM1_OVF_vect)
			
	#endif
	
	#ifdef _IAR_C
		#include <ioavr.h>
		#include <inavr.h>
		#include <pgmspace.h>
		
		/*!	\def _SIN_TABLE(i)
		 *	\brief Lookup sine table.
		 *
		 *	Lookup the sine table stored in flash memory and access the ith element
		 *	
		 */
		#define _SIN_TABLE(i)						sinewave[i]
	
		/*!	\def _COS_TABLE(i)
		 *	\brief Lookup cos table.
		 *
		 *	Lookup the same sine table as in <I>#define _SIN_TABLE(i)</I> stored 
		 *	in flash memory,convert it into an effective cos table and retrieve
		 *	the value of the ith element in the effective cos table.	
		 */
		#define _COS_TABLE(i)						sinewave[(i+64)%256]
	
		/*!	\def _TIMER1_OVERFLOW_INTERRUPT()
		 *	\brief Compiler specific header for timer1 overflow ISR
		 *
		 *	This statement is neccecary for incorporating both AVR_GCC, and IAR_C
		 *	compilers. Defines a common 'TIMER1 overflow interrupt' header.
		 */
		#define _TIMER1_OVERFLOW_INTERRUPT()		__interrupt void Timer1_Overflow_ISR(void)
		
	#endif
	
	#include "motor.h"
	#include "config.h"
	#include "tables.h"
	#include "uart.h"
	#include "motor.c"
	#include "uart.c"
	
#endif
